-- Aufträge
CREATE TABLE auftg (
  ag_id                SERIAL PRIMARY KEY,
  ag_astat             VARCHAR(1) NOT NULL,
  ag_nr                VARCHAR(30) NOT NULL CONSTRAINT xtt16539__ag_nr CHECK (strpos(ag_nr, '%') = 0),
  ag_stat              VARCHAR(40), -- Pos.Typ / Sorte / Warenwirtschaftsstatus
                                    -- [BL = Beistellung Lieferant (Interner Auftrag)]

  ag_pos               INTEGER NOT NULL,
  ag_hpos              INTEGER CONSTRAINT xtt11104 CHECK(ag_hpos IS DISTINCT FROM ag_pos), -- Übergeordnete Position (Hauptpos), diese Position ist deren Unterposition. Constraint="Position kann keine untergeordnete Position von sich selbst sein."
  ag_nstatistik        BOOL NOT NULL DEFAULT FALSE,    -- nicht in Statistik, nicht umsatzwirksam
  ag_lkn               VARCHAR(21) NOT NULL REFERENCES adk ON UPDATE CASCADE,           -- besteller
  ag_krzl              VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, -- lieferadresse
  ag_krzf              VARCHAR(30) NOT NULL REFERENCES adressen_keys ON UPDATE CASCADE, -- rechnungsadresse
  ag_bda               VARCHAR(40), -- Bestellnummer d. Kunden
  ag_bdapos            INTEGER,     -- Bestellposition d. Kunden
  ag_ang_ag_id         INTEGER,     -- angebots ag_id bei Angebot->Auftrag
  ag_kanf_nr           VARCHAR(20), -- REFERENCES kundanfrage,          -- wird in X_TableConstraints erstellt
  ag_vtp_id            INTEGER,     -- REFERENCES in X_TableConstraints -- VertragsID
  ag_dispokrzl         VARCHAR(10),
  ag_dispo             VARCHAR(50),                            -- Ansprechpartner Kunde
  ag_kontakt           VARCHAR(30) DEFAULT tsystem.current_user_ll_db_usename(), -- Zuständiger Mitarbeiter
  ag_nident            VARCHAR(50) REFERENCES qsnorm,          -- Qualitätsnorm
  ag_bdat              DATE DEFAULT current_date,
  ag_aknr              VARCHAR(40) NOT NULL REFERENCES art ON UPDATE CASCADE,
  ag_aknr_idx          VARCHAR(40),                            -- Zeichnungsindex, vgl. art.ak_idx
  ag_akbz              VARCHAR(100),
  ag_aknr_referenz     VARCHAR(50),                            -- Artikelnummer Kunde
  --
  ag_stk               NUMERIC(14,6) NOT NULL,                 -- In der Oberfläche = Menge Verkauf
  ag_nk_stk            NUMERIC(14,6),
  ag_stk_uf1           NUMERIC(20,8),
  ag_stkl              NUMERIC(20,8) NOT NULL DEFAULT 0,       -- gelieferte Menge
  ag_nk_stkl           NUMERIC(20,8),                          -- Menge verbraucht (gebucht)
  ag_nk_stkl_uf1       NUMERIC(20,8),
  ag_nk_w_wen          INTEGER,                                -- Wareneingangsnummer für Rohmaterial in Bearbeiten NK
  ag_stkb              NUMERIC(20,8) NOT NULL DEFAULT 0,       -- Menge bestellt (eingekauft)
  ag_stkf              NUMERIC(20,8) NOT NULL DEFAULT 0,       -- verrechnete Menge
  ag_stkres            NUMERIC(12,4) NOT NULL DEFAULT 0,
  ag_stkres_uf1        NUMERIC(20,8),
  ag_mcv               INTEGER NOT NULL REFERENCES artmgc,
  ag_nbedarf           BOOLEAN NOT NULL DEFAULT FALSE,
  ag_sperr             BOOLEAN NOT NULL DEFAULT FALSE,                          -- Sperrstatus #9197
  ag_datum             DATE DEFAULT current_date,         -- Erfassungsdatum des Auftrags
  ag_arab              NUMERIC(5,2) NOT NULL DEFAULT 0,   -- der Rabatt der nicht NIL sein, da sonst die Berechnung Auftragssumme mist macht
  ag_rabhint           VARCHAR(75),
  --
  ag_preis             NUMERIC(12,4)  NOT NULL,           -- Neu ab 01/2016, Name analog Belegpos und Reporting Views, Feldlänge wie ag_vkp bisher. http://redmine.prodat-sql.de/issues/6447
  ag_preiseinheit      NUMERIC(12,4)  NOT NULL DEFAULT 1 CONSTRAINT auftg__chk__preiseinheit CHECK ( ag_preiseinheit > 0), -- Bezugsgröße von ag_preis,
  ag_vkp               NUMERIC(20,8) NOT NULL,            --internes Feld, verrechnet mit Preiseinheit
  ag_vkptotalpos       BOOL NOT NULL DEFAULT FALSE,       -- Position enthält den Preis aller Unterpositionen
  ag_vkp_uf1           NUMERIC(20,8),
  ag_vkp_basis_w       NUMERIC(20,8),
  ag_vkp_uf1_basis_w   NUMERIC(20,8),
  ag_nk_vkp            NUMERIC(16,4),                     -- Preis in der Nachkalkulation: für Material, hierhinein schreibt auch Zubuchen der Eingangsrechnung. Der Kopfartikel-Gesamtpreis der Nachkalkulation steht in abk.ab_nk_et
  ag_nk_vkp_uf1        NUMERIC(20,8),
  ag_nk_calc_vkp       NUMERIC(16,4),                     -- kalkulierter Durchschnittspreis anhand realen Buchungen
  ag_nk_calc_vkp_uf1   NUMERIC(20,8),
  ag_waer              VARCHAR(3) NOT NULL /*CONSTRAINT xtt4056*/ REFERENCES bewa,
  ag_vkpbas            NUMERIC(12,4),
  ag_vkp_mce           INTEGER REFERENCES artmgc,                               -- #7326 Verkaufs - Preismengeneinheit
  --
  ag_netto             NUMERIC(16,4),          -- Positionswert inkl. AbZuschläge
  ag_ep_netto          NUMERIC(16,4),          -- Einzelpreis ohne Abzuschläge inkl. Rabatte
  ag_brutto            NUMERIC(16,4),          -- Positionswert inkl. AbZuschläge,Steuern
  ag_netto_basis_w     NUMERIC(16,4),          -- Positionswert inkl. AbZuschläge in Basiswährung
  ag_brutto_basis_w    NUMERIC(16,4),          -- Positionswert inkl. AbZuschläge,Steuern in Basiswährung
  --
  ag_hest              NUMERIC(12,2),
  ag_rust              NUMERIC(12,2),
  ag_fert              NUMERIC(12,2),
  ag_mat               NUMERIC(12,2),
  ag_awkost            NUMERIC(12,2),
  ag_ks                VARCHAR(9) REFERENCES ksv,
  ag_konto             VARCHAR(12),
  ag_post1             VARCHAR(150),                   -- Kundenspezifische Textfelder (Kundtxt!)
  ag_post2             VARCHAR(150),                   -- bei internen Aufträgen(ag_astat='i') ist das Feld der Materialtyp
  ag_post3             VARCHAR(150),                   -- bei internen Aufträgen(ag_astat='i') ist das Feld gleich ag_parentabk
  ag_post4             VARCHAR(150),
  ag_post5             VARCHAR(150),
  ag_post6             VARCHAR(150),
  ag_post7             VARCHAR(150),
  ag_kdatum            DATE,                           -- Wunschdat (Kunde) / Bedarfsdatum
  ag_ldatum            DATE,                           -- Liefdat best. (an Kunde)
  ag_aldatum           DATE,                           -- Auslieferdatum damit Auftrag pünktlich beim Kunden ankommt
  ag_termv             DATE,                           -- Datum Auftrag, verschoben
  ag_ldatumtext        VARCHAR(150),                   -- Liefertermin frei
  ag_twa               VARCHAR(7),                     -- Kalenderwoche Termweek
  ag_steucode          INTEGER REFERENCES steutxt,     -- Neu (Nov.2012) Steuercode des Auftrags, bisher nur ustpr (Prozentsatz)
  ag_ustpr             NUMERIC(5,2),
  ag_prkl              INTEGER NOT NULL,
  ag_kukl              INTEGER NOT NULL,
  ag_bstat             VARCHAR(2), -- REFERENCES auftgbs,  --Statusflags für Auftrag  -- wird in X_TableConstraints erstellt
  ag_bstat1            VARCHAR(2), -- REFERENCES auftgbs,  --Statusflags für Auftrag  -- wird in X_TableConstraints erstellt
  ag_bstat2            VARCHAR(2), -- REFERENCES auftgbs,  --Statusflags für Auftrag  -- wird in X_TableConstraints erstellt
  ag_kurs              NUMERIC(12,4) NOT NULL,
  ag_txt               TEXT,
  ag_txt_rtf           TEXT,
  ag_postxt            TEXT,
  ag_postxt_rtf        TEXT,
  ag_dokunr            INTEGER,
  ag_done              BOOLEAN NOT NULL DEFAULT FALSE,
  ag_storno            BOOLEAN NOT NULL DEFAULT FALSE,   -- Wenn true, auftrag storniert => ag_done auf true
  ag_o6_dimi           VARCHAR(50),
  ag_o6_stkz           INTEGER,        -- Stück  Zuschnitt
  ag_o6_lz             NUMERIC(12,4),  -- Länge  Zuschnitt
  ag_o6_bz             NUMERIC(12,4),  -- Breite Zuschnitt
  ag_o6_hz             NUMERIC(12,4),  -- Höhe   Zuschnitt
  ag_o6_zz             NUMERIC(12,4),  -- Zugabe Zuschnitt
  ag_o6_zme            INTEGER REFERENCES mgcode, -- ZuschnittME
  ag_o6_pos            INTEGER,        -- Pos Zuschnitt #10848
  ag_hwpos             VARCHAR(15),
  ag_alt               VARCHAR(5),     -- Kennzeichen zur Steuerung
  ag_an_nr             VARCHAR(50) REFERENCES anl ON UPDATE CASCADE,
  ag_stornodat         DATE,
  ag_canRabatt         BOOLEAN NOT NULL DEFAULT TRUE, -- Posistionsartikel ist rabattfähig
  ag_bem               TEXT,
  ag_parentabk         INTEGER,
  ag_ownabk            INTEGER,
  ag_mainabk           INTEGER,
  ag_a2_id             INTEGER,               -- REFERENCES ab2 ON UPDATE CASCADE ON DELETE SET NULL > Foreign Key in Table Constraints
  ag_q_nr              INTEGER,               -- REFERENCES qab, -- wird in X_TableConstraints erstellt, enthält den QAB, für den dieser Auftrag ggf. als Nacharbeitsauftrag erstellt wurde
  ag_rahmen_ag_id      INTEGER REFERENCES auftg ON UPDATE CASCADE,              -- verknüpfter Rahmen
  ag_kanfwsk_wsk       VARCHAR(20), -- REFERENCES kanfwsk ON UPDATE CASCADE     -- #6679 Erweiterung Interessentenverwaltung, Referenz später, siehe  X_TableConstraints
  ag_absgrup           VARCHAR(20) --- REFERENCES kundabs ON UPDATE CASCADE -- #11689 Absage-Grund-Gruppe
  );

 -- auftg Indizes
    CREATE UNIQUE INDEX xtt5107 ON auftg(ag_astat, ag_nr, ag_pos);

    CREATE INDEX auftg_ag_nr ON auftg(ag_nr varchar_pattern_ops);

    CREATE INDEX auftg_ag_aknr ON auftg(ag_aknr varchar_pattern_ops);

    CREATE INDEX auftg_ag_lkn ON auftg(ag_lkn varchar_pattern_ops);
    CREATE INDEX auftg_ag_krzl ON auftg(ag_krzl varchar_pattern_ops);
    CREATE INDEX auftg_ag_krzf ON auftg(ag_krzf varchar_pattern_ops);

    CREATE INDEX auftg_is_done ON auftg(ag_id) WHERE ag_done;
    CREATE INDEX auftg_isnt_done ON auftg(ag_id) WHERE NOT ag_done;

    CREATE INDEX auftg_dokunr ON auftg(ag_dokunr) WHERE ag_dokunr IS NOT NULL;
    CREATE INDEX auftg_dokunr_text ON auftg(CAST(ag_dokunr AS TEXT) varchar_pattern_ops) WHERE CAST(ag_dokunr AS TEXT) IS NOT NULL;

    CREATE INDEX auftg_ag_ang_ag_id ON auftg(ag_ang_ag_id) WHERE ag_ang_ag_id IS NOT NULL;

    CREATE INDEX auftg_agmainabk        ON auftg(ag_mainabk)                                        INCLUDE ( ag_id ) WHERE ag_mainabk IS NOT NULL;
    CREATE INDEX auftg_agmainabk_text   ON auftg(CAST(ag_mainabk AS TEXT) varchar_pattern_ops)      INCLUDE ( ag_id ) WHERE CAST(ag_mainabk AS TEXT) IS NOT NULL;
    CREATE INDEX auftg_agownabk         ON auftg(ag_ownabk)                                         INCLUDE ( ag_id ) WHERE ag_ownabk IS NOT NULL;
    CREATE INDEX auftg_agownabk_text    ON auftg(CAST(ag_ownabk AS TEXT) varchar_pattern_ops)       INCLUDE ( ag_id ) WHERE CAST(ag_ownabk AS TEXT) IS NOT NULL;
    CREATE INDEX auftg_agparentabk      ON auftg(ag_parentabk)                                      INCLUDE ( ag_id ) WHERE ag_parentabk IS NOT NULL;
    CREATE INDEX auftg_agparentabk_text ON auftg(CAST(ag_parentabk AS TEXT) varchar_pattern_ops)    INCLUDE ( ag_id ) WHERE CAST(ag_parentabk AS TEXT) IS NOT NULL;

    CREATE INDEX auftg_rahmen           ON auftg(ag_rahmen_ag_id)                                   INCLUDE ( ag_id ) WHERE ag_rahmen_ag_id IS NOT NULL;
    CREATE INDEX auftg_ag_a2_id         ON auftg(ag_a2_id)                                          INCLUDE ( ag_id ) WHERE ag_a2_id IS NOT null;
    
    CREATE INDEX auftg_ag_an_nr ON auftg(ag_an_nr varchar_pattern_ops) WHERE ag_an_nr IS NOT NULL;
    CREATE INDEX auftg_ag_bda ON auftg(AEOEUE_UPPER(ag_bda) varchar_pattern_ops) WHERE AEOEUE_UPPER(ag_bda) IS NOT NULL;
    CREATE INDEX auftg_ag_kanf_nr ON auftg(ag_kanf_nr varchar_pattern_ops) WHERE ag_kanf_nr IS NOT NULL;
    CREATE INDEX auftg_ag_q_nr ON auftg(ag_q_nr) WHERE ag_q_nr IS NOT NULL;
    CREATE INDEX auftg_ag_q_nr_text ON auftg(CAST(ag_q_nr AS TEXT) varchar_pattern_ops) WHERE CAST(ag_q_nr AS TEXT) IS NOT NULL;

    CREATE INDEX auftg_bdat ON auftg(ag_bdat);
    CREATE INDEX auftg_ag_ldatum ON auftg(ag_ldatum);
    CREATE INDEX auftg_ag_datum ON auftg(ag_datum);
    CREATE INDEX auftg_ag_kl_datum ON auftg(COALESCE(ag_ldatum,ag_kdatum));

    CREATE INDEX auftg_ym_ldatum ON auftg(date_to_yearmonth(ag_ldatum));
    CREATE INDEX auftg_ym_term ON auftg(date_to_yearmonth(COALESCE(ag_ldatum, ag_kdatum)));
    CREATE INDEX auftg_ym_datum ON auftg(date_to_yearmonth(ag_datum));

    CREATE INDEX auftg_agpost1 ON auftg(AEOEUE_UPPER(ag_post1) varchar_pattern_ops) WHERE AEOEUE_UPPER(ag_post1) IS NOT NULL;
    CREATE INDEX auftg_agpost2 ON auftg(AEOEUE_UPPER(ag_post2) varchar_pattern_ops) WHERE AEOEUE_UPPER(ag_post2) IS NOT NULL;
    CREATE INDEX auftg_agpost3 ON auftg(AEOEUE_UPPER(ag_post3) varchar_pattern_ops) WHERE AEOEUE_UPPER(ag_post3) IS NOT NULL;
    CREATE INDEX auftg_agpost4 ON auftg(AEOEUE_UPPER(ag_post4) varchar_pattern_ops) WHERE AEOEUE_UPPER(ag_post4) IS NOT NULL;
    CREATE INDEX auftg_agpost5 ON auftg(AEOEUE_UPPER(ag_post5) varchar_pattern_ops) WHERE AEOEUE_UPPER(ag_post5) IS NOT NULL;
    CREATE INDEX auftg_agpost6 ON auftg(AEOEUE_UPPER(ag_post6) varchar_pattern_ops) WHERE AEOEUE_UPPER(ag_post6) IS NOT NULL;
 --
 ALTER TABLE auftg ADD CONSTRAINT auftg__hpos_subpos_exists_cant_delete FOREIGN KEY (ag_astat, ag_nr, ag_hpos) REFERENCES auftg( ag_astat, ag_nr, ag_pos) ON UPDATE CASCADE;

  -- Diverse Defaults und Vorbereitungen. Läuft vor den spezielleren auftg_b_iu_xyz-Triggern.
 CREATE OR REPLACE FUNCTION auftg__b_01_iu() RETURNS TRIGGER AS $$
     DECLARE r    record;
     BEGIN
         IF (new.ag_pos = 0 OR new.ag_astat = 'R') AND new.ag_rahmen_ag_id IS NOT NULL THEN
             RAISE EXCEPTION '% %', lang_text(16307), new.ag_nr; -- Ein Rahmenauftrag darf selbst nicht mit einem Rahmenauftrag verknüpft sein.
         END IF;
         -- Vorprüfungen und Defaults zuerst, da die Werte weiter unten ggf. schon verwendet werden.
         new.ag_astat        := upper(new.ag_astat);
         new.ag_nr           := upper(new.ag_nr);
         new.ag_preis        := coalesce(new.ag_preis, 0);
         new.ag_preiseinheit := coalesce(new.ag_preiseinheit, 1);
         IF NOT  new.ag_done 
            AND  (   ( (TG_OP = 'INSERT') AND (new.ag_kurs IS null) ) -- kein Kurs durch Oberfläche angegeben. Kurs aus Währung holen
                  OR ( (TG_OP = 'UPDATE') AND (new.ag_waer IS DISTINCT FROM old.ag_waer) AND (new.ag_kurs IS NOT DISTINCT FROM old.ag_kurs) ) -- Wechsel der Währung, aber Kurs NICHT verändert. Kurs aktualisieren!
                 )
         THEN      --- #19766 Wechselkur soll auch neue geholt werden
             new.ag_kurs     := coalesce( waerkurs( new.ag_waer ), 1 );
         END IF;
         -- WaWi-Felder nie unter 0. Fall: Bestellung löschen, wo mehr produziert wird, als Bedarf hier war (zB Sollmenge).
         new.ag_stkb         := greatest(new.ag_stkb, 0);
         new.ag_stkl         := greatest(new.ag_stkl, 0);
         new.ag_stkf         := greatest(new.ag_stkf, 0);
         --
         IF TG_OP = 'INSERT' THEN -- vorher per auftg__b_10_i. Auflösung aufgrund Abhängigkeiten in auftg__b_05_iu__stkvkpuf1, siehe #11971.

             -- Nachsetzen von Adressdaten, falls nur erste angegeben.
             new.ag_krzl     := coalesce(new.ag_krzl, new.ag_lkn);
             new.ag_krzf     := coalesce(new.ag_krzf, new.ag_lkn);
             --
             -- Defaults für Materialpositionen
             IF new.ag_astat = 'I' THEN
                 new.ag_prkl := 0;
                 new.ag_kukl := 0;
                 new.ag_lkn  := '#';
                 new.ag_krzl := TAdk.get_vorg_ladress('#');
                 new.ag_krzf := '#';
                 -- Termin und Projektnummer aus übergeordneter ABK holen
                 IF new.ag_parentabk IS NOT NULL THEN -- Interne Aufträge haben eigentlich immer eine parentabk
                     SELECT ab_at, ab_tablename, ab_dbrid, ab_keyvalue, ab_an_nr INTO r FROM abk LEFT JOIN anl ON an_ab_ix = ab_ix WHERE ab_ix = new.ag_parentabk;

                     -- Interne Auftragsnummernvergabe
                     IF new.ag_nr IS NULL THEN
                         new.ag_nr  := TWaWi.auftgi__ag_nr__Generate(new.ag_parentabk, new.ag_mainabk, r.ab_tablename, r.ab_dbrid, r.ab_keyvalue);
                         new.ag_pos := TWaWi.auftg__ag_pos__next(new.ag_astat, new.ag_nr);
                     END IF;

                     -- Übernahme Norm aus Artikel ODER Haupt-Bestellung
                     SELECT coalesce(ld_nident, ak_norm) INTO new.ag_nident FROM art LEFT JOIN ldsdok ON ld_abk = new.ag_mainabk WHERE ak_nr = new.ag_aknr;

                     new.ag_kdatum := COALESCE(r.ab_at, new.ag_kdatum); -- Interne MatPosition: Notwendiges Datum aus ABK übernehmen
                     new.ag_an_nr  := r.ab_an_nr;
                 END IF;
             END IF;
         END IF;

         -- Positionen automatisch ziehen
         IF new.ag_pos IS NULL AND new.ag_nr IS NOT NULL THEN
             new.ag_pos := TWaWi.auftg__ag_pos__next(new.ag_astat, new.ag_nr);
         END IF;
         -- Preis und Preiseinheit
             IF TG_OP = 'INSERT' THEN
                 -- Kein manueller Preis angegeben, ag_vkp von irgendwoher gefüllt -> ag_preis zurückrechnen.
                 IF new.ag_preis = 0 AND new.ag_vkp <> 0 THEN
                     new.ag_preis := new.ag_vkp * new.ag_preiseinheit;
                 ELSE -- ag_preis angegeben oder alles ist 0
                     new.ag_vkp   := new.ag_preis / Do1If0(new.ag_preiseinheit);
                 END IF;
             END IF;

             IF TG_OP = 'UPDATE' THEN
                 -- Preis / PE geändert -> VKP Updaten
                 IF ((new.ag_preis IS DISTINCT FROM old.ag_preis) OR (new.ag_preiseinheit IS DISTINCT FROM old.ag_preiseinheit)) THEN
                     new.ag_vkp   := new.ag_preis / Do1If0(new.ag_preiseinheit);
                 ELSIF new.ag_vkp <> old.ag_vkp THEN -- VKP geändert und Preis / PE gleich -> Preis zurückrechnen.
                     new.ag_preis := new.ag_vkp * new.ag_preiseinheit;
                 END IF;
             END IF;
         --
         -- Storno und Stornodatum
             -- Stornieren beendet Auftrag und setzt Stornodatum
             IF new.ag_storno THEN
                 new.ag_done := true;
                 IF new.ag_stornodat IS NULL THEN
                     new.ag_stornodat := current_date;
                 END IF;
             END IF;

             -- Stornierung aufheben, Stornodatum entfernen. Auftrag bleibt aber erledigt.
             IF TG_OP = 'UPDATE' THEN
                 IF NOT new.ag_storno AND old.ag_storno THEN
                     new.ag_stornodat := NULL;
                 END IF;
             END IF;

             -- Stornodatum setzen setzt auch storniert Flag.
             IF new.ag_stornodat IS NOT NULL AND NOT new.ag_storno THEN
                 new.ag_storno := true;
             END IF;
         --
         RETURN new;
     END $$ LANGUAGE plpgsql;

     CREATE TRIGGER auftg__b_01_iu
       BEFORE UPDATE OR INSERT
       ON auftg
       FOR EACH ROW
       EXECUTE PROCEDURE auftg__b_01_iu();
 --
 --Änderung am Artikel
 CREATE OR REPLACE FUNCTION auftg__b_01_iu__aknr() RETURNS TRIGGER AS $$
    DECLARE artr RECORD;
    BEGIN
    -- Preise und Kosten laut Artikelstamm
        SELECT ak_vkpbas, ak_hest, ak_rust, ak_fertk, ak_matk, ak_awkost, ak_pknr,
               -- Preisfaktoren für nicht interne Aufträge berücksichtigen.
               -- Auf Angeboten Kostenbestandteile mit Faktoren ausweisen, #9323 (JM).
               CASE WHEN new.ag_astat = 'I' THEN 1 -- Für Materialposition gibt es keine Faktoren.
               ELSE COALESCE(NullIf(ak_vkpfaktor, 0), NullIf(ac_vkpfaktor, 0), 1) -- Bei allen an den Kunden gerichteten Vorgängen Faktoren berücksichtigen.
               END AS faktor
          INTO artr
          FROM art
          JOIN artcod ON ac_n = ak_ac
         WHERE ak_nr = new.ag_aknr;
        --
        new.ag_vkpbas := artr.ak_vkpbas;
        new.ag_hest   := artr.ak_hest;
        --
        new.ag_rust   := artr.ak_rust   * artr.faktor;
        new.ag_fert   := artr.ak_fertk  * artr.faktor;
        new.ag_mat    := artr.ak_matk   * artr.faktor;
        new.ag_awkost := artr.ak_awkost * artr.faktor;


        -- INSERT
        IF new.ag_mcv IS NULL THEN --Standard ME für Artikel
           new.ag_mcv := tartikel.me__art__artmgc__m_id__by__ak_standard_mgc(new.ag_aknr);
        END IF;        
        --
        IF tg_op = 'UPDATE' THEN
           --Artikelnummer darf für interne Aufträge nicht umgeschrieben werden, wenn schon geliefert oder verrechnet wurde.
           IF (new.ag_astat = 'I') AND (new.ag_pos > 0) AND (old.ag_stkl + old.ag_stkf <> 0) THEN
               RAISE EXCEPTION 'xtt11560'; -- Für Kundenaufträge zugelassen, damit man Artikel nochmal anpassen kann, wenn versehentlich schon Rechnung gestellt wurde.
           END IF;
           --
           -- die meid hat sich nicht geändert, obwohl sich der Artikel geändert hat!
           IF (new.ag_aknr IS DISTINCT FROM old.ag_aknr) AND (new.ag_mcv IS NOT DISTINCT FROM old.ag_mcv)
           THEN /*wir müssen natürlich auch den Mengencode mit nachziehen!*/
              new.ag_mcv := tartikel.me__convertme_for_art__by__mid(new.ag_aknr, old.ag_mcv, true);
           END IF;
           IF (new.ag_aknr IS DISTINCT FROM old.ag_aknr) AND (new.ag_vkp_mce IS NOT null) AND (new.ag_vkp_mce IS NOT DISTINCT FROM old.ag_vkp_mce)
           THEN /*wir müssen natürlich auch den Mengencode mit nachziehen!*/
              new.ag_vkp_mce := tartikel.me__convertme_for_art__by__mid(new.ag_aknr, old.ag_vkp_mce, false);
           END IF;           
        END IF;
        --

        -- Der Datensatz wurde mit einer für den Artikel ungültigen Mengeneinheit gespeichert. Tabelle: %, ID: %, Feld: %, ART-Nr.: %, Artmgc-ID: %
        IF NOT TArtikel.me__art__artmgc__m_ids__valid( new.ag_aknr, new.ag_mcv )
        THEN
            RAISE EXCEPTION     '%', Format(lang_text(13795), 'auftg', new.ag_id::VARCHAR, 'ag_mcv'    , new.ag_aknr, new.ag_mcv::VARCHAR);
        END IF;

        IF NOT TArtikel.me__art__artmgc__m_ids__valid( new.ag_aknr, new.ag_vkp_mce )
        THEN -- Preismengeneinheit
           RAISE EXCEPTION '%', Format(lang_text(13795), 'auftg', new.ag_id::VARCHAR, 'ag_vkp_mce', new.ag_aknr, new.ag_vkp_mce::VARCHAR);
        END IF;
        --
        RETURN new;
    END $$ LANGUAGE plpgsql;

    --CREATE TRIGGER auftg__b_u
    CREATE TRIGGER auftg__b_01_iu__aknr
      BEFORE INSERT OR UPDATE
      OF ag_aknr, ag_mcv, ag_vkp_mce
      ON auftg
      FOR EACH ROW
      WHEN (new.ag_aknr IS NOT null) -- wenn null, kommt NOT NULL CONSTRAINT
      EXECUTE PROCEDURE auftg__b_01_iu__aknr();

 -- Normierung auf UF1
 CREATE OR REPLACE FUNCTION auftg__b_05_iu__stkvkpuf1() RETURNS TRIGGER AS $$ --beachte ldsdok__b_05_iu__stkvkpuf1
    BEGIN
        -- Zuschnitte
        IF new.ag_o6_stkz + new.ag_o6_lz > 0 THEN -- Wenn Zuschnitte verwendet werden, kann Menge nicht einfach angepasst werden
            IF TG_OP = 'UPDATE' THEN
                IF old.ag_stk <> new.ag_stk THEN
                    RAISE EXCEPTION 'xtt17421'; -- Menge konnte nicht verändert werden, da Zuschnitte existieren. (ag_stk überschrieben durch Berechnungsergebnis aus dem Zuschnitt)
                END IF;
            END IF;

            IF new.ag_o6_zme IS NOT NULL THEN
                new.ag_stk := tartikel.zuschnittmenge(new.ag_o6_lz,
                                                      new.ag_o6_bz,
                                                      new.ag_o6_hz,
                                                      new.ag_o6_zz,
                                                      Round(new.ag_o6_stkz)::INTEGER,
                                                      TArtikel.me__art__artmgc__m_ids__uf(new.ag_o6_zme, TArtikel.me__mec__by__mid(new.ag_mcv)));
            ELSE
                new.ag_stk := tartikel.zuschnittmenge(new.ag_o6_lz, new.ag_o6_bz, new.ag_o6_hz, new.ag_o6_zz, new.ag_o6_stkz);
            END IF;
        END IF;
    --
    new.ag_stk_uf1         := tartikel.me__menge__in__menge_uf1(new.ag_mcv, new.ag_stk);
    new.ag_stkres_uf1      := tartikel.me__menge__in__menge_uf1(new.ag_mcv, new.ag_stkres);

    -- Defaults für Materialpositionen
    IF TG_OP = 'INSERT' AND new.ag_astat = 'I' THEN -- vorher per auftg__b_10_i. Auflösung aufgrund Abhängigkeiten in auftg__b_05_iu__stkvkpuf1, siehe #11971.
        new.ag_preis := COALESCE(NullIf(new.ag_preis, 0), new.ag_hest * new.ag_stk_uf1 / Do1If0(new.ag_stk), 0); -- Initialer Preis ist von ag_stk_uf1 abhängig. Ermittlung erst hier möglich.
        new.ag_vkp   := new.ag_preis; -- Hier notwendig, da UF1-Feld bei initalem Preis für MatPos gefüllt werden muss.
    END IF;

    new.ag_vkp_uf1         := tartikel.me__preis__in__preis_uf1(COALESCE(new.ag_vkp_mce, new.ag_mcv), new.ag_vkp); -- #7326
    new.ag_vkp_basis_w     := new.ag_vkp * new.ag_kurs;
    new.ag_vkp_uf1_basis_w := new.ag_vkp_uf1 * new.ag_kurs;
    -- Nachkalkulation
    new.ag_nk_vkp_uf1      := tartikel.me__preis__in__preis_uf1(COALESCE(new.ag_vkp_mce, new.ag_mcv), new.ag_nk_vkp);
    new.ag_nk_stkl_uf1     := tartikel.me__menge__in__menge_uf1(new.ag_mcv, new.ag_nk_stkl);
    new.ag_nk_calc_vkp     := tartikel.me__preis_uf1__in__preis(new.ag_mcv, new.ag_nk_calc_vkp_uf1); -- #8308, Durchschnittspreis wird in UF1 geschrieben. Daher hier Wert in UF schreiben.

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_05_iu__stkvkpuf1 --Beachte ldsdok, EPreis usw
    BEFORE INSERT OR UPDATE
    OF ag_stk, ag_nk_stkl, ag_stkres
       , ag_mcv, ag_vkp_mce  --hinweis: aknr nicht nötig, da Trigger aknr bei Änderung Artikel ohne Änderung ME automatisch die neue ME wählt
       , ag_vkp, ag_preis, ag_preiseinheit, ag_kurs, ag_waer --todo waer
       , ag_o6_lz, ag_o6_bz, ag_o6_hz, ag_o6_zz, ag_o6_stkz
       , ag_nk_vkp, ag_nk_calc_vkp_uf1 --nur auftg, alles andere auch ldsdok
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__b_05_iu__stkvkpuf1();
 --

 --Terminwoche, Datum verschoben etc
 CREATE OR REPLACE FUNCTION auftg__b_iu__dat() RETURNS TRIGGER AS $$
  DECLARE do_change_ag_aldatum BOOL;
  BEGIN
    IF coalesce(new.ag_ldatum, new.ag_kdatum)<current_date-(365*100) THEN --datum liegt 100 Jahre zurück => Schwachsinn
        RAISE EXCEPTION 'auftg__b_iu__dat: invalid date: %', coalesce(new.ag_ldatum, new.ag_kdatum);
    END IF;

    IF new.ag_kdatum IS NOT NULL AND new.ag_ldatum IS NULL THEN--Terminwoche setzen
        new.ag_twa := week_of_year(new.ag_kdatum);
    END IF;
    IF new.ag_ldatum IS NOT NULL THEN--Terminwoche setzen
        new.ag_twa := week_of_year(new.ag_ldatum);
    END IF;

    -- Auslieferdatum gibt es bei Intern und Rahmen nicht
    IF new.ag_astat IN ('I', 'R') THEN
        new.ag_aldatum := null;
        new.ag_termv := null;
        RETURN new;
    END IF;

    -- hier bereits !!!!! EXIT wenn I und R

    --Liefertermin bestätigt geändert, auslieferdatum anpassen, wenn nicht bereits aus oberfläche angepasst
    do_change_ag_aldatum := False;
    IF tg_op = 'INSERT' THEN
        do_change_ag_aldatum := coalesce(new.ag_ldatum, new.ag_kdatum) IS NOT NULL AND new.ag_aldatum IS NULL;
    ELSE
        IF new.ag_aldatum IS NOT DISTINCT FROM old.ag_aldatum THEN 
           --NICHT in Oberfläche geändert, dann verändern wir es aufgrund der Änderung des Liefertermin
            do_change_ag_aldatum :=         (  (new.ag_ldatum IS NULL AND new.ag_kdatum IS DISTINCT FROM old.ag_kdatum) 
                                             OR(new.ag_ldatum IS DISTINCT FROM old.ag_ldatum) 
                                             )
                                    AND NOT (new.ag_aldatum IS DISTINCT FROM old.ag_aldatum)
                                    AND     coalesce(new.ag_ldatum, new.ag_kdatum) IS NOT NULL ;
        END IF;
    END IF;
    --
    IF do_change_ag_aldatum THEN
        new.ag_aldatum :=       timediff_addsubstdays(coalesce(new.ag_ldatum, new.ag_kdatum), -(coalesce(ad_ldauer,0) + IFTHEN(TSystem.Settings__GetBool('auftg_alterm_inkl_tol'), a1_tolfrueh, 0)), true) 
                           FROM adressen_view LEFT JOIN adk1 ON a1_krz = adk_ad_krz 
                          WHERE ad_krz = new.ag_krzl;
    END IF;
    --
    IF tg_op = 'UPDATE' THEN
        IF new.ag_termv IS NULL AND new.ag_ldatum IS DISTINCT FROM old.ag_ldatum THEN
            new.ag_termv := old.ag_ldatum;
        END IF;
        IF new.ag_termv = new.ag_ldatum THEN
            new.ag_termv := NULL;
        END IF;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__dat
    BEFORE INSERT OR UPDATE
    OF ag_ldatum, ag_kdatum, ag_termv
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__b_iu__dat();
 --

--
CREATE OR REPLACE FUNCTION auftg__b_iu__aknr__stati() RETURNS TRIGGER AS $$
  DECLARE
      _ak_idx   varchar;
      _ac_i     integer;
  BEGIN
    -- Ermittelt und setzt
      -- Zeichnungsindex zum Zeitunkt der Erstellung und Änderung des Artikels
      -- Materialtyp zum Zeitunkt der Erstellung sowie Änderung des Artikels und ABK-Bezuges


    -- IC und Artikel-Revision/Zeichnungsindex ermitteln.
      -- fraglich: Kann Status ak_fertigung auch berücksichtigt werden (z.B. Kaufteile die auch gefertigt werden).
    SELECT
      ak_idx,
      ac_i
    FROM art
      JOIN artcod ON ak_ac = ac_n
    WHERE ak_nr = new.ag_aknr
    INTO
      _ak_idx,
      _ac_i
    ;


    -- Zeichnungsindex in Materialposition ablegen
    IF tg_op = 'INSERT' THEN
        -- Zeichnungsindex bei Erstellung übertragen.
        -- abweichende Anwendereingaben erlauben
        IF new.ag_aknr_idx IS NULL AND _ak_idx IS NOT NULL THEN
            new.ag_aknr_idx := _ak_idx;
        END IF;

    -- UPDATE
    ELSE
        -- Bei Änderung des Artikels den Zeichnungsindex aktualisieren.
        -- abweichende Anwendereingaben erlauben (Update auf ag_aknr_idx behandelt Trigger nicht).
        IF new.ag_aknr <> old.ag_aknr AND new.ag_aknr_idx IS NOT DISTINCT FROM old.ag_aknr_idx THEN
            new.ag_aknr_idx := _ak_idx;
        END IF;

    END IF;


    -- Ermittlung der Materialtypen:
    -- IC 10 oder Bezug zu anderer Fertigung => Einzel- oder Montageteil (Eigene Produkte bzw. Vorprodukte)
    IF ( _ac_i = 10 OR new.ag_ownabk IS NOT NULL ) THEN

        -- Montageteil (hat Unterbauteile)
        IF  (
              -- Stücklisten-Element vorhanden (insb. wenn bei IC 10 ag_ownabk nicht gesetzt ist)
                  EXISTS(SELECT true FROM stv   WHERE st_zn = new.ag_aknr)
              -- Unterbauteil in eigener Fertigung vorhanden (insb. wenn ag_ownabk gesetzt und IC <> 10 ist)
              OR  EXISTS(SELECT true FROM auftg WHERE ag_parentabk = new.ag_ownabk AND ag_post2 IS DISTINCT FROM 'R')
            )
        THEN
            new.ag_post2 := 'M';

        -- Einzelteil (ohne Unterbauteile)
        ELSE
            new.ag_post2 := 'E';

        END IF;

    -- IC 3 => Rohmaterial
    ELSIF _ac_i = 3 THEN
        new.ag_post2 := 'R';

    -- alle anderen IC oder kein Bezug zu anderer Fertigung => Norm- bzw. Kaufteil
    ELSE
        new.ag_post2 := 'N';

    END IF;


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__aknr__stati
    BEFORE INSERT OR UPDATE
      -- ag_ownabk und ag_post4 aktualisieren sich wechselseitig, s.o. Es muss also auf beides reagiert werden.
    OF ag_aknr, ag_post2, ag_ownabk, ag_post4
    ON auftg
    FOR EACH ROW
    WHEN ( new.ag_astat = 'I' )
    EXECUTE PROCEDURE auftg__b_iu__aknr__stati();
--

--
CREATE OR REPLACE FUNCTION auftg__a_iu__agpost2() RETURNS TRIGGER AS $$
  BEGIN

    -- Falls wir z.B. woanders druntergeschoben wurden, müssen die Materialpositionen, die diese ABK verwenden,
    -- auch neu validiert werden. Ggf. sind diese jetzt Materialtyp M.
    IF new.ag_parentabk IS NOT NULL THEN

        -- rekursiv ag_post2 füllen
        UPDATE auftg SET ag_post2 = 'X' WHERE ag_ownabk = new.ag_parentabk AND ag_post2 IS DISTINCT FROM 'R';

    END IF;


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu__agpost2
    AFTER INSERT OR UPDATE
      -- ag_parentabk und ag_post3 aktualisieren sich wechselseitig, s.o. Es muss also auf beides reagiert werden.
    OF ag_aknr, ag_post2, ag_parentabk, ag_post3
    ON auftg
    FOR EACH ROW
    WHEN ( new.ag_astat = 'I' )
    EXECUTE PROCEDURE auftg__a_iu__agpost2();
--

-- Bezug auf eine übergeordnete Position vorhanden. Aktuelle Position ist also eine Unterposition.
CREATE OR REPLACE FUNCTION auftg__b_iu__hpos() RETURNS TRIGGER AS $$
  DECLARE
        mainposstorno           bool;
        _parent_vkptotalpos_id  integer;
  BEGIN
    -- Umnummerieren: Positionen werden *-1 gesetzt.
    IF new.ag_hpos < 0 THEN
        RETURN new;
    END IF;

    IF tg_op = 'UPDATE' THEN
        IF old.ag_hpos < 0 THEN
            RETURN new;
        END IF;
    END IF;
    --

    -- ID der obersten Position holen, wenn das eine vkptotalpos ist.
    _parent_vkptotalpos_id := auftg_get_mainpos( new, true );

    -- Wenn Hauptposition storniert wurde, die aktuelle Position auch stornieren.
    mainposstorno := ag_storno FROM auftg WHERE ag_id = _parent_vkptotalpos_id AND ag_vkptotalpos;

    IF mainposstorno THEN
        new.ag_storno := true;
    END IF;
    --

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__hpos
    BEFORE INSERT OR UPDATE
    OF ag_hpos
    ON auftg
    FOR EACH ROW
    WHEN ( new.ag_hpos IS NOT NULL )
    EXECUTE PROCEDURE auftg__b_iu__hpos();
--

-- Positionsswert neu berechnen, wird auch von auftgabzu__a_iud aufgerufen, indem ag_netto auf 0 gesetzt wird.
CREATE OR REPLACE FUNCTION auftg__b_50_iu__possum() RETURNS TRIGGER AS $$
  DECLARE
        rec      record;
        mainagid integer;
        faktor   numeric;
        gesrab   numeric;
  BEGIN
    -- Umnummerieren: Positionen werden *-1 gesetzt.
    IF new.ag_hpos < 0 THEN
        RETURN new;
    END IF;

    IF tg_op = 'UPDATE' THEN
        IF old.ag_hpos < 0 THEN
            RETURN new;
        END IF;
    END IF;
    --

    -- rabattierten Preis berechnen
    -- Position ist rabattfähig? Dann Faktor ermitteln aus Positions- und Gesamtrabatt.
    IF new.ag_canRabatt THEN
        -- Von aktueller Position schauen wir ganz nach oben, ob irgendwo drüber eine vkptotalpos ist.
        -- Wenn das der Fall ist, haben wir von dieser den gesrab zu nehmen.
        mainagid := auftg_get_mainpos( new, true, true);

        -- Wir sind selbst "Preis enthält unterposition" ODER haben keine Hauptpos ORDER es gibt keine VKPTOTALPOS-Hauptpos.
        IF mainagid IS NULL AND new.ag_hpos IS NULL THEN -- ich bin oberste wurzel
            gesrab := atd_gesrab FROM auftgdokutxt WHERE atd_dokunr = new.ag_dokunr;
        END IF;
        --
        IF mainagid IS NOT NULL THEN
            gesrab := atd_gesrab FROM auftgdokutxt WHERE atd_dokunr = ( SELECT ag_dokunr FROM auftg WHERE ag_id = mainagid );
        END IF;
        --

        faktor := ( 1- (new.ag_arab / 100)) * ( 1 - COALESCE( gesrab, 0 ) / 100 );
    ELSE
        -- Artikel kann keinen Rabatt erhalten
        faktor      := 1;
        new.ag_arab := 0;
    END IF;
    --

    new.ag_ep_netto := new.ag_vkp * faktor;

    -- prozentuale Abzu neu berechnen
    UPDATE auftgabzu SET
      -- verwendet rabatierten Preis
      az_abzubetrag = coalesce(new.ag_ep_netto * az_abzuproz / 100, 0)
    WHERE az_ag_id = new.ag_id
      AND az_abzuproz IS NOT NULL
      -- Rundungsabweichungen einbeziehen, NUMERIC(12,4)
      AND az_abzubetrag <> round(coalesce(new.ag_ep_netto * az_abzuproz/100, 0), 4)
    ;

    -- Abzüge/Zuschläge = (Anzahl * Betrag * [Steuerprozent])
    SELECT
      sum(coalesce(az_abzubetrag, 0) * az_anz) AS abzunetto,
      sum(coalesce(az_abzubetrag, 0) * az_anz * (1 + coalesce(az_steuproz, 0) / 100)) AS abzubrutto
    INTO rec
    FROM auftgabzu
    WHERE az_ag_id = new.ag_id
    ;

    -- UF1 Werte in auftg__b_05_iu__stkvkpuf1
    -- Positionssummen, da ag_vkp schon durch auftg__b_iu aktualisiert wurde, sind hier ag_preis und ag_preiseinheit bereits enhalten.
    new.ag_netto          := (new.ag_stk_uf1 * new.ag_vkp_uf1 * faktor) + coalesce(rec.abzunetto, 0);
    new.ag_brutto         := (new.ag_stk_uf1 * new.ag_vkp_uf1 * faktor) * (1 + coalesce(new.ag_ustpr, 0) / 100) + coalesce(rec.abzubrutto, 0);
    new.ag_netto_basis_w  := new.ag_netto   * new.ag_kurs;
    new.ag_brutto_basis_w := new.ag_brutto  * new.ag_kurs;
    --

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_50_iu__possum -- vgl. ldsdok__b_50_iu__possum
    BEFORE INSERT OR UPDATE
    OF  ag_stk, ag_vkp, ag_preis, ag_arab, ag_canrabatt, ag_kurs, ag_ustpr, ag_netto,
        ag_mcv, ag_vkp_mce, ag_preiseinheit,
        ag_hpos,
        ag_o6_lz, ag_o6_bz, ag_o6_hz, ag_o6_zz, ag_o6_stkz
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__b_50_iu__possum();
--

 --
 CREATE OR REPLACE FUNCTION auftg__b_75_iu__bstat_k() RETURNS TRIGGER AS $$
   BEGIN
    -- ACHTUNG, wenn KONSIGNATIONSPROZESS, dann ist der Auftrag nicht umsatzwirksam #7780
    IF (NOT TSystem.Settings__GetBool('EntnahmeVomKonsiLiefsch')) THEN
      -- wenn nicht vom Lieferschein entnommen werden soll (verschlankter Konsi-Prozess) #7943
      new.ag_nstatistik := TRUE;
    END IF;
    RETURN new;
   END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_75_iu__bstat_k
   BEFORE INSERT OR UPDATE
   OF ag_bstat, ag_bstat1, ag_bstat2
   ON auftg
   FOR EACH ROW
   WHEN ((new.ag_astat = 'E') AND ('K' IN (new.ag_bstat, new.ag_bstat1, new.ag_bstat2)))
   EXECUTE PROCEDURE auftg__b_75_iu__bstat_k();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__b_iu__sync_agpost_abk() RETURNS TRIGGER AS $$
  BEGIN
    --RAISE NOTICE 'sync_agpost_abk:%,%,%;%,%,%', new.ag_post3, new.ag_post6, new.ag_post5, new.ag_parentabk, new.ag_mainabk, new.ag_ownabk;
    --Problemfall: einfügen feuert immer, dadurch wurde ag_parentabk entfernt, bevor es in ag_post schreiben konnte
    IF tg_op='UPDATE' THEN--das Update hier geschieht nur, wenn direkt in einem der AG_POST Felder geändert wurde. Falls etwas entfernt wurde, vorher zurücksetzen.
       new.ag_parentabk := NULL;--Diesen Mistcode mal entfernen und alle Oberfläche&Bearbeitungsfelder direkt auf die ag_parent, ...main und ...ownabk zugreifen lassen
       new.ag_ownabk    := NULL;
       new.ag_mainabk   := NULL;
    END IF;
    --übergeordnete ABK = die, wo wir eingehen
    IF coalesce(new.ag_post3, '') <> '' THEN
       new.ag_parentabk := CAST(new.ag_post3 AS integer);
    END IF;
    --eigene ABK = die, mittels welche wir selbst produziert werden. somit : auftg -> ldsdok -> abk <-> eigeneabk
    IF coalesce(new.ag_post4, '') <> '' THEN
        new.ag_ownabk := CAST(new.ag_post4 AS integer);
    END IF;
    --hauptABK bei Struktur/Baugruppe
    IF coalesce(new.ag_post6, '') <> '' THEN
        new.ag_mainabk := CAST(new.ag_post6 AS integer);
    END IF;
    --
    IF new.ag_mainabk IS NULL THEN
       new.ag_mainabk := tplanterm.abk_main_abk(new.ag_parentabk);
       new.ag_post6   := new.ag_mainabk;
    END IF;
    --RAISE NOTICE 'sync_agpost_abk:%,%,%;%,%,%', new.ag_post3, new.ag_post6, new.ag_post5, new.ag_parentabk, new.ag_mainabk, new.ag_ownabk;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__0__sync_agpost_abk
    BEFORE INSERT OR UPDATE OF ag_post3, ag_post4, ag_post6
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_astat = 'I')
    EXECUTE PROCEDURE auftg__b_iu__sync_agpost_abk();
 --
 CREATE OR REPLACE FUNCTION auftg__b_iu__sync_abk_agpost() RETURNS TRIGGER AS $$
  BEGIN
    --trigger, falls direkt ag_parent/mainabk etc geändert wurden, ohne das die ag_post geändert wurden
    --RAISE NOTICE 'sync_abk_agpost:%,%,%;%,%,%', new.ag_post3, new.ag_post6, new.ag_post5, new.ag_parentabk, new.ag_mainabk, new.ag_ownabk;
    IF new.ag_post3 IS DISTINCT FROM new.ag_parentabk THEN
        new.ag_post3 := new.ag_parentabk;
    END IF;
    --
    IF new.ag_post4 IS DISTINCT FROM new.ag_ownabk THEN
        new.ag_post4 := new.ag_ownabk;
    END IF;
    --
    IF new.ag_post6 IS DISTINCT FROM new.ag_mainabk THEN
        new.ag_post6 := new.ag_mainabk;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__1__sync_abk_agpost
    BEFORE INSERT OR UPDATE OF ag_parentabk, ag_mainabk, ag_ownabk
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_astat = 'I')
    EXECUTE PROCEDURE auftg__b_iu__sync_abk_agpost();
  --  

CREATE OR REPLACE FUNCTION auftg__b_iu__ag_ownabk__check() RETURNS TRIGGER AS $$
  DECLARE
    _agid_dup integer;
  _agnr_dup varchar; 
  _agpos_dup integer;
  BEGIN
  
    -- #20138 Eine ABK darf nicht zwei verschiedenen Aufträgen zugeordnet werden, zur Vermeidung von Endlosschleifen.

    _agid_dup := ag_id FROM auftg WHERE ag_ownabk = new.ag_ownabk AND ag_id <> new.ag_id;
    IF _agid_dup IS NOT null THEN  
      SELECT ag_nr, ag_pos INTO _agnr_dup, _agpos_dup FROM auftg WHERE ag_id = _agid_dup;
      RAISE EXCEPTION 'ABK % wurde mehreren Aufträgen zugeordnet: (%, %), (%, %) xtt28705', new.ag_ownabk, new.ag_nr, new.ag_pos, _agnr_dup, _agpos_dup;  
    END IF;

    RETURN new;
  
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__ag_ownabk__check
    BEFORE INSERT OR UPDATE OF ag_ownabk
    ON auftg
    FOR EACH ROW
    WHEN ( new.ag_ownabk IS NOT null )
    EXECUTE PROCEDURE auftg__b_iu__ag_ownabk__check();
  --    

 CREATE OR REPLACE FUNCTION auftg__a_iu__2__sync__ab_parentabk() RETURNS TRIGGER AS $$
  BEGIN    --falls wir eine eigene ABK haben, müssen die Parents auch übereinstimmen
    UPDATE abk
       SET ab_parentabk = new.ag_parentabk
     WHERE ab_ix = new.ag_ownabk
       AND new.ag_parentabk IS DISTINCT FROM ab_parentabk;
    -- ABK setzt ggf ag_mainabk
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu__2__sync__ab_parentabk
    AFTER INSERT OR UPDATE OF ag_post3, ag_post4, ag_post6, ag_parentabk, ag_mainabk, ag_ownabk
    ON auftg
    FOR EACH ROW
    WHEN ( (new.ag_astat = 'I') AND (new.ag_ownabk IS NOT null) )
    EXECUTE PROCEDURE auftg__a_iu__2__sync__ab_parentabk();
 --

 -- automatische Angebotsannahme und Verlinkung der Angebotsdokumente in Auftrag
 CREATE OR REPLACE FUNCTION auftg__b_iu__ag_ang_ag_id() RETURNS TRIGGER AS $$
  DECLARE do_link BOOLEAN;
          ang_rec RECORD;
  BEGIN
    -- UPDATE OF ag_ang_ag_id, ag_lkn, ag_aknr, ag_datum
    -- WHEN (new.ag_q_nr IS NULL AND new.ag_nr NOT LIKE '.%QS')

    -- automatische Angebotsannahme
    IF TSystem.Settings__GetBool('LinkAngAuftr') THEN
        do_link:= false;

        IF new.ag_astat = 'E' AND new.ag_ang_ag_id IS NULL THEN
            IF TG_OP = 'INSERT' THEN -- INSERT-Fall: Bei leerer Angebotsreferenz immer durchführen
                do_link:= true;
            ELSIF old.ag_ang_ag_id IS NULL THEN -- UPDATE-Fall: Nur durchführen, wenn Angebotsreferenz noch nicht gefüllt (nicht beim Entfernen der Angebotsreferenz), nur bei Änderungen der anderen Felder.
                do_link:= true;
            END IF;

            IF do_link THEN -- automatische Angebotsannahme durchführen
                SELECT NOT ag_done AS lastangoffen, ag_id AS lastangid, 'Angebot angenommen: ' || ag_nr || ' Pos.:' || ag_pos || ' vom ' || ag_datum AS msg
                INTO ang_rec
                FROM auftg WHERE ag_astat = 'A' AND ag_lkn = new.ag_lkn AND ag_aknr = new.ag_aknr AND ag_datum <= new.ag_datum ORDER BY ag_datum DESC;

                IF ang_rec.lastangoffen THEN -- wenn letztes Angebot offen
                    PERFORM PRODAT_TEXT(ang_rec.msg);
                    new.ag_ang_ag_id:= ang_rec.lastangid;
                END IF;
            END IF;
        END IF;
    END IF;

    do_link:= false; -- Verlinkung der Angebotsdokumente durchführen

    IF new.ag_ang_ag_id IS NOT NULL THEN
        IF TG_OP = 'INSERT' THEN -- INSERT-Fall: Immer durchführen, wenn Angebotsreferenz vorhanden ist.
            do_link:= true;
        ELSIF old.ag_ang_ag_id IS DISTINCT FROM new.ag_ang_ag_id THEN -- UPDATE-Fall: Nur durchführen, wenn Angebotsreferenz sich geändert hat.
            do_link:= true;
        END IF;

        IF do_link THEN
            UPDATE auftg SET ag_done= true WHERE ag_id = new.ag_ang_ag_id AND NOT ag_done; -- Angebotsposition erfüllt setzen, wenn Zuordnung Angebot/Auftrag

            PERFORM CreatePicnDokuLink(pd_id, 'auftgtxt', auftgtxt.dbrid) -- Angebotsdokumente in Auftrag verlinken
            FROM picndoku
              JOIN auftgtxt ON at_astat = new.ag_astat AND at_nr = new.ag_nr
            WHERE pd_tablename = 'auftgtxt' AND pd_doktype = 'aufadok'
              AND pd_dbrid IN (SELECT dbrid FROM auftgtxt WHERE at_astat = 'A' AND at_nr = (SELECT ag_nr FROM auftg WHERE ag_id = new.ag_ang_ag_id))
              AND NOT EXISTS(SELECT true FROM picndokulink WHERE pdl_pd_id = pd_id AND pdl_tablename = 'auftgtxt' AND pdl_dbrid = auftgtxt.dbrid);
        END IF;
    END IF;

    -- Verlinkung der Angebotsdokumente bei Änderung wieder entfernen
    IF TG_OP = 'UPDATE' THEN
        IF old.ag_ang_ag_id IS NOT NULL AND old.ag_ang_ag_id IS DISTINCT FROM new.ag_ang_ag_id THEN
            DELETE FROM picndokulink
            WHERE pdl_tablename = 'auftgtxt'
              AND pdl_dbrid IN (SELECT dbrid FROM auftgtxt WHERE at_astat = old.ag_astat AND at_nr = old.ag_nr)
              AND pdl_pd_id IN (SELECT pd_id FROM picndoku
                                WHERE pd_tablename = 'auftgtxt' AND pd_doktype = 'aufadok'
                                  AND pd_dbrid IN (SELECT dbrid FROM auftgtxt WHERE at_astat = 'A' AND at_nr = (SELECT ag_nr FROM auftg WHERE ag_id = old.ag_ang_ag_id)));
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

   CREATE TRIGGER auftg__b_iu__ag_ang_ag_id
    BEFORE INSERT OR UPDATE
    OF ag_ang_ag_id, ag_lkn, ag_aknr, ag_datum
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_q_nr IS NULL AND new.ag_nr NOT LIKE '.%QS' AND new.ag_rahmen_ag_id IS NULL) --https://redmine.prodat-sql.de/issues/10083
    EXECUTE PROCEDURE auftg__b_iu__ag_ang_ag_id();
 --

 -- Externen Text an Lieferschein- und Rechnungspositionen weitergeben, wenn dort noch nicht geändert und nicht definitv/gedruckt. Ansonsten Meldung.
 CREATE OR REPLACE FUNCTION auftg__a_u__ag_txt() RETURNS TRIGGER AS $$
  DECLARE infotxt TEXT;
  BEGIN -- WHEN (old.ag_txt IS DISTINCT FROM new.ag_txt)
    -- Meldung für nicht veränderbare Lieferscheinpositionen, bei denen der externe Text schon unterschiedlich ist oder das Dokument definitiv bzw. gedruckt.
    IF EXISTS(SELECT true FROM lieferschein_pos JOIN lieferschein ON belp_dokument_id = beld_id WHERE beld_belegtyp = 'LFS' AND belp_ag_id = new.ag_id AND belp_dokutxt IS DISTINCT FROM new.ag_txt AND (belp_dokutxt IS DISTINCT FROM old.ag_txt OR beld_definitiv OR beld_print)) THEN
        infotxt:=lang_text(16340)||E'\n\n'||lang_text(527)||E':\n'|| -- Der Dokumenttext in den Folgedokumenten wurde nicht angepasst ... Lieferschein: ...
            (SELECT array_to_string(ARRAY( -- Stringliste aller Lieferscheinpositionen
                SELECT beld_dokunr||' '||lang_text(164)||' '||belp_pos
                FROM lieferschein_pos JOIN lieferschein ON belp_dokument_id = beld_id
                WHERE beld_belegtyp = 'LFS' AND belp_ag_id = new.ag_id
                  AND belp_dokutxt IS DISTINCT FROM new.ag_txt -- Ziel soll geändert werden
                  AND (belp_dokutxt IS DISTINCT FROM old.ag_txt OR beld_definitiv OR beld_print) -- war aber vorher abweichend, definitiv oder gedruckt
                ORDER BY beld_dokunr, belp_pos
                ), E'\n'));
    END IF;


    -- In Rechnung soll immer Text aus Artikelstamm genommen werden #8296
    IF  NOT TSystem.Settings__GetBool('bzUseArtTxt') THEN
      -- Meldung für nicht veränderbare Rechnungspositionen, bei denen der externe Text schon unterschiedlich ist oder das Dokument definitiv.
      IF EXISTS(SELECT true FROM belzeil_grund JOIN belkopf ON be_bnr = bz_be_bnr WHERE bz_auftg = new.ag_nr AND bz_auftgpos = new.ag_pos AND bz_zubez IS DISTINCT FROM new.ag_txt AND (bz_zubez IS DISTINCT FROM old.ag_txt OR be_def)) THEN
        IF infotxt IS NULL THEN
            infotxt:=lang_text(16340);
        END IF;
        infotxt:=infotxt||E'\n\n'||lang_text(379)||E':\n'|| -- ... Faktura: ...
            (SELECT array_to_string(ARRAY( -- Stringliste aller Rechnungspostionen
                SELECT be_prof||' '||bz_be_bnr||' '||lang_text(164)||' '||bz_pos
                FROM belzeil_grund JOIN belkopf ON be_bnr = bz_be_bnr
                WHERE bz_auftg = new.ag_nr AND bz_auftgpos = new.ag_pos
                  AND bz_zubez IS DISTINCT FROM new.ag_txt -- Ziel soll geändert werden
                  AND (bz_zubez IS DISTINCT FROM old.ag_txt OR be_def) -- war aber vorher abweichend oder definitiv
                ORDER BY be_prof, bz_be_bnr, bz_pos
                ), E'\n'));
      END IF;
    END IF;

    IF infotxt IS NOT NULL THEN
        PERFORM PRODAT_TEXT(infotxt);
    END IF;

    -- In Rechnung soll immer Text aus Artikelstamm genommen werden #8296
    IF  NOT TSystem.Settings__GetBool('bzUseArtTxt') THEN
      -- Rechnungspositionen zuerst updaten, weil Änderung auch im Lieferschein, siehe belegpos__a_u__belp_dokutxt
      UPDATE belzeil_grund SET
        bz_zubez = new.ag_txt,
        bz_zubez_rtf = new.ag_txt_rtf
      FROM belkopf
      WHERE bz_be_bnr = be_bnr
        AND NOT be_def
        AND bz_auftg = new.ag_nr AND bz_auftgpos = new.ag_pos
        AND bz_zubez IS NOT DISTINCT FROM old.ag_txt -- waren die Texte vorher gleich
        AND bz_zubez IS DISTINCT FROM new.ag_txt;
    END IF;

    -- Lieferscheinpositionen, die automatisch synchron gehalten werden.
    UPDATE lieferschein_pos SET
      belp_dokutxt = new.ag_txt,
      belp_dokutxt_rtf = new.ag_txt_rtf
    FROM lieferschein
    WHERE belp_dokument_id = beld_id AND beld_belegtyp = 'LFS'
      AND NOT beld_definitiv AND NOT beld_print
      AND belp_ag_id = new.ag_id
      AND belp_dokutxt IS NOT DISTINCT FROM old.ag_txt -- waren die Texte vorher gleich
      AND belp_dokutxt IS DISTINCT FROM new.ag_txt;

    RETURN new;
  END $$ LANGUAGE plpgsql;

   CREATE TRIGGER auftg__a_u__ag_txt
    AFTER UPDATE
    OF ag_txt
    ON auftg
    FOR EACH ROW
    WHEN (old.ag_txt IS DISTINCT FROM new.ag_txt)
    EXECUTE PROCEDURE auftg__a_u__ag_txt();
 --

 -- ag_parentabk, ag_mainabk bei internen Bestellungen nicht leer #2601
 -- _ziu wegen alphanumerischer Auflösung der Triggerreihenfolge erforderlich, muss zum Schluss kommen.
 CREATE OR REPLACE FUNCTION auftg__a_ziu__obligate_abk() RETURNS TRIGGER AS $$
  BEGIN
    IF EXISTS(SELECT TRUE FROM auftg WHERE ag_id = new.ag_id ) THEN -- wegen ABK-Löschen und DEFERRED Trigger-Prüfung, innnerhalb Transaktion Felder SET NULL, dann DELETE auftg
        RAISE EXCEPTION '%', lang_text(16180)||E'\n\n'||'Auftrag: I '||new.ag_nr||E'\n'||'Pos: '||new.ag_pos;
    END IF;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE CONSTRAINT TRIGGER auftg__a_ziu__obligate_abk
    AFTER INSERT OR UPDATE OF ag_parentabk, ag_mainabk, ag_storno
    ON auftg
    DEFERRABLE INITIALLY DEFERRED
    FOR EACH ROW
    WHEN (new.ag_astat='I' AND (NOT new.ag_storno) AND (new.ag_parentabk IS NULL OR new.ag_mainabk IS NULL))
    EXECUTE PROCEDURE auftg__a_ziu__obligate_abk();
 --

 -- Verfügbarkeit in Art setzen
 CREATE OR REPLACE FUNCTION auftg__a_10_i() RETURNS TRIGGER AS $$
  DECLARE rows INTEGER;
          beistell BOOLEAN;
  BEGIN
    IF current_user='syncro' THEN
        RETURN new;
    END IF;

    --kopfdatensatz anlegen
    IF NOT EXISTS(SELECT true FROM auftgtxt WHERE at_astat=new.ag_astat AND at_nr=new.ag_nr) THEN
        INSERT INTO auftgtxt (at_astat, at_nr) VALUES (new.ag_astat, new.ag_nr);
    END IF;

    IF new.ag_astat = 'I' THEN

       -- wir fügen Material unter einer ABK ein. War diese bereits geschlossen, öffnen wir diese wieder https://redmine.prodat-sql.de/issues/18683
       IF NOT new.ag_done THEN
          UPDATE abk SET ab_done = false
           WHERE ab_ix = new.ag_parentabk AND ab_done;
       END IF;

       --bei hydro immer alle termine mit dem kundenbestätigungstermin nachziehen
       IF TSystem.Settings__Get('KUNDE') IN ('HYDRO', 'MEYER') THEN
           SELECT ag_ldatum
             INTO new.ag_ldatum
             FROM auftg
            WHERE ag_id = (SELECT ld_ag_id FROM ldsdok WHERE ld_abk = new.ag_mainabk);--lieferdatum aus ursprungsauftrag holen
       END IF;
    END IF;

    -- Bei SUN sind im Verkaufsauftrag die Materialien nie Bedarfswirksam, erst die interne Materialliste.
    IF new.ag_astat = 'E' THEN
       beistell := TSystem.Settings__Get('KUNDE') IN ('SUNSTROM', 'MR-SUN') ;
    END IF;
    --

    INSERT INTO auftgmatinfo(agmi_ag_id, agmi_beistell) VALUES (new.ag_id, COALESCE(beistell, false));--Datensatz für Detailinfo anlegen. Bei Sunstrom ist Externauftrag immer als Beistellung zu betrachten
    --dadurch wir automatisch ein richtiger bestandsabgleich auf ak_res gemacht!
    --In PRODAT wird immer zuerst das auftg geposted, dadurch wird der insert trigger hier IMMER den auftgmatinfo satz anlegen!

    --IF new.ag_astat='E' THEN
    --   INSERT INTO auftgliefplanung (aglp_ag_id) VALUES (new.aglp_ag_id);
    --END IF;

    PERFORM tartikel.prepare_artikel_bedarf(new.ag_aknr, true);
    RETURN new;
  END $$ LANGUAGE plpgsql;
  --
  CREATE TRIGGER auftg__a_10_i
    AFTER INSERT
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_10_i();
 --
 CREATE OR REPLACE FUNCTION auftg__a_12_iu() RETURNS TRIGGER AS $$
  DECLARE
    vkpupos                   NUMERIC(16,4);
    anzupos                   NUMERIC(16,4);
    dokugesamt                NUMERIC(16,4);
    abzunetto                 NUMERIC(16,4);
    mainagid                  INTEGER;
    vkptotalpos               BOOL;
    hposid                    BOOL;
    mainposisopt              BOOL;
    stornopos                 BOOL;
    rundabweichung            NUMERIC(16,4);
    _hauptpositions_dokunr    int;
  BEGIN

      IF auftgtriggerdisabled() THEN
          RETURN new;
      END IF;

      IF ( public.auftg_get_mainpos( new , true, true ) IS NOT NULL ) THEN

        _hauptpositions_dokunr := ag_dokunr from auftg where ag_id = public.auftg_get_mainpos( new , true, true );

        IF ( new.ag_dokunr IS NOT NULL AND _hauptpositions_dokunr is distinct from new.ag_dokunr ) THEN
          RAISE EXCEPTION ' cannot mix ag_dokunr within a postree ';
        END IF;

      END IF;

      -- muß wegen datenübernahme schon hier stehen (transferdata)
      IF new.ag_dokunr IS NOT NULL AND NOT EXISTS( SELECT true FROM auftgdokutxt WHERE atd_dokunr = new.ag_dokunr ) THEN

           INSERT INTO auftgdokutxt ( atd_dokunr, atd_apkrzl, atd_ap, atd_apint )
           VALUES (
             new.ag_dokunr,
             new.ag_dispokrzl,
             new.ag_dispo,
              CASE WHEN TSystem.Settings__GetBool('dok_ap_is_current_user')
                THEN tsystem.current_user_ll_db_usename()
                ELSE COALESCE( new.ag_kontakt, tsystem.current_user_ll_db_usename() ) END
            );

      END IF;
      --
      IF ( current_user = 'syncro' ) THEN
          RETURN new;
      END IF;

      --
      -- an_lief
      -- Änderung Materialliste
      IF ( tg_op = 'INSERT' ) AND ( new.ag_astat = 'I' ) THEN -- hinzufügen von Material
          UPDATE abk SET ab_matchange=TRUE WHERE ab_ix=new.ag_parentabk AND ab_print AND NOT ab_matchange;
      END IF;

      IF ( tg_op = 'UPDATE' ) AND ( new.ag_astat = 'I' ) THEN

          -- Änderung - nicht durchs Lager (ausliefern darf nicht gleichzeitig als Änderung zählen)
          IF new.ag_stkl=old.ag_stkl THEN
              UPDATE abk SET ab_matchange=TRUE WHERE ab_ix = new.ag_parentabk AND ab_print AND NOT ab_matchange;
          END IF;

      END IF;

      --Aktualisieren der ProjektNr in Kundenanfrage und Erfassungsbogen / Kundenanfrage-Nr. in Projektverwaltung
      IF ( new.ag_an_nr IS NOT NULL ) AND ( new.ag_kanf_nr IS NOT NULL ) THEN
          UPDATE kundanfrage SET kanf_an_nr = new.ag_an_nr WHERE kanf_nr = new.ag_kanf_nr AND kanf_an_nr IS DISTINCT FROM new.ag_an_nr;
      END IF;

      -- Meldung ausgeben,wennn Steuersatz ungültig weil abgelaufen
      IF EXISTS(  SELECT true FROM steutxt WHERE steu_z = new.ag_steucode AND steu_valid_to < current_date ) THEN
          -- Leider keine Verknüpfung über steucode möglich
          PERFORM PRODAT_TEXT(15810);
      END IF;

      --
      RETURN new;

  END $$ LANGUAGE plpgsql;
  --
  CREATE TRIGGER auftg__a_12_iu
    AFTER UPDATE OR INSERT
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_12_iu();
 --
 CREATE OR REPLACE FUNCTION auftg__a_14_u() RETURNS TRIGGER AS $$
  DECLARE rec RECORD;
          abix INTEGER;
          dokugesamt NUMERIC(16,4);
  BEGIN
    IF (current_user='syncro') THEN
        RETURN new;
    END IF;

    IF auftgtriggerdisabled() THEN -- FUNCTION disableauftgtrigger()
        RETURN new;
    END IF;

    IF       new.ag_datum < current_date AND new.ag_astat IN ('E' , 'R')
       AND ((new.ag_netto <> old.ag_netto) OR (new.ag_storno <> old.ag_storno) OR (new.ag_nstatistik <> old.ag_nstatistik)) 
    THEN
        PERFORM enablemodified();--insert by usw in log führen
        INSERT INTO auftglog (agl_ag_id, agl_old_ag_stk, agl_new_ag_stk, agl_old_ag_netto, agl_new_ag_netto)
               VALUES (new.ag_id, old.ag_stk, new.ag_stk, IFTHEN(old.ag_storno OR old.ag_nstatistik, 0, old.ag_netto), IFTHEN(new.ag_storno OR new.ag_nstatistik, 0, new.ag_netto));
        PERFORM disablemodified();
    END IF;

    /*Position abgeschlossen, keine Statistik und Verfügbarkeiten mehr vergeben*/
    IF old.ag_done AND new.ag_done THEN
        RETURN new;
    END IF;

    IF (old.ag_astat <> new.ag_astat) OR (old.ag_nr <> new.ag_nr) 
    THEN/*Angebot in Auftrag*/
        IF NOT EXISTS(SELECT true FROM auftg WHERE ag_astat = old.ag_astat AND ag_nr = old.ag_nr) 
        THEN --Zusatztexte zum Auftrag!
            UPDATE auftgtxt 
               SET at_astat = new.ag_astat, at_nr = new.ag_nr 
             WHERE at_astat = old.ag_astat 
               AND at_nr = old.ag_nr 
               AND NOT EXISTS(SELECT true FROM auftgtxt WHERE at_astat = new.ag_astat AND at_nr = new.ag_nr)/*letzter Part: Umschreiben der Auftragsnummer bei mehreren Positionen*/;
        ELSE--einzelne Position übernommen, Text kopieren
            INSERT INTO auftgtxt (at_astat, at_nr, at_txt, at_txt_rtf/*, at_gesrab*/) 
                 SELECT new.ag_astat, new.ag_nr, at_txt, at_txt_rtf/*, at_gesrab*/ 
                   FROM auftgtxt 
                  WHERE at_astat = old.ag_astat 
                    AND at_nr = old.ag_nr;
        END IF;
    END IF;

    IF NOT EXISTS(SELECT true FROM auftg WHERE ag_astat = old.ag_astat AND ag_nr = old.ag_nr) 
    THEN --Zusatztexte zum Auftrag!
        DELETE FROM auftgtxt WHERE at_astat = old.ag_astat AND at_nr = old.ag_nr;
    END IF;

    IF new.ag_dokunr IS NULL AND old.ag_dokunr IS NOT NULL 
    THEN
        IF NOT EXISTS(SELECT true FROM auftg WHERE ag_dokunr = old.ag_dokunr) 
        THEN --Zusatztexte zum Auftrag (Dokument)!
            DELETE FROM auftgdokutxt WHERE atd_dokunr = old.ag_dokunr;
        END IF;
    END IF;

    -- #7686
    IF NOT Equals(new.ag_kanf_nr, old.ag_kanf_nr) THEN
        IF new.ag_kanf_nr IS NULL THEN -- Kundenanfrage entfernen
            UPDATE kundanfragepos 
               SET kanfp_angnr = NULL, kanfp_ag_id = NULL 
             WHERE kanfp_ag_id = old.ag_id; -- alten Bezug entfernen
        ELSE                -- Kundenanfrage ändern
            UPDATE kundanfragepos 
               SET kanfp_angnr = NULL, kanfp_ag_id = NULL 
             WHERE kanfp_ag_id = old.ag_id; -- alten Bezug entfernen
            UPDATE kundanfragepos 
               SET kanfp_angnr = new.ag_nr, kanfp_ag_id = new.ag_id -- neuen herstellen (ggf. auch mehrfach bei gleichem Artikel)
             WHERE kanfp_kanf_nr = new.ag_kanf_nr 
               AND kanfp_ak_nr = new.ag_aknr 
               AND kanfp_angnr IS NULL;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;
  --
  CREATE TRIGGER auftg__a_14_u
    AFTER UPDATE
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_14_u();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_i__rahmenab() RETURNS TRIGGER AS $$
  BEGIN
    -- Rahmen schließen aufgrund Abrufmenge
    UPDATE auftg SET
      ag_done=TRUE
    WHERE ag_id=new.ag_rahmen_ag_id
      AND NOT ag_done
      AND tauftg.auftgr__rahmen_info__stko_offen__by__ag_id__get(new.ag_rahmen_ag_id)<=0;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_i__rahmenab
    AFTER INSERT
    ON auftg
    FOR EACH ROW
    WHEN ((new.ag_rahmen_ag_id IS NOT NULL) AND (NOT new.ag_storno))
    EXECUTE PROCEDURE auftg__a_i__rahmenab();
 --

-- Position verschieben, wenn Childs, dann dort die Nummer mit ändern.
-- Löst evtl. Trigger ag_hpos_change aus, daher abfangen.
CREATE OR REPLACE FUNCTION auftg__a_11_u__agpos() RETURNS TRIGGER AS $$
  BEGIN
    -- Positionsnummer ändern, Unterpositionen mitziehen.
    PERFORM disablemodified();
    PERFORM disableauftgtrigger();

    UPDATE auftg SET
      ag_hpos = new.ag_pos
    WHERE ag_hpos = old.ag_pos
      AND ag_astat = new.ag_astat
      AND ag_nr = new.ag_nr
    ;

    PERFORM enablemodified();
    PERFORM enableauftgtrigger();

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_11_u__agpos
    AFTER UPDATE
    OF ag_pos
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_11_u__agpos();
--

 --
 CREATE OR REPLACE FUNCTION auftg__a_u__lifschupdate() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM disablemodified();
    UPDATE lifsch SET l_krzl=new.ag_krzl, l_krzf=new.ag_krzf, l_an_nr=new.ag_an_nr WHERE l_ag_id=new.ag_id;
    PERFORM enablemodified();
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_u__lifschupdate
    AFTER UPDATE
    OF ag_krzl, ag_krzf, ag_an_nr
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_u__lifschupdate();
 --

 -- Mengen des Rahmens mit Abrufen synchronisieren
 CREATE OR REPLACE FUNCTION auftg__a_u__rahmenauftgupdate() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (COALESCE(new.ag_rahmen_ag_id, old.ag_rahmen_ag_id) IS NOT NULL) --nur für Abrufe
    PERFORM disablemodified();

    -- Wenn Rahmen aus Abruf entfernt oder Abruf storniert wird, gelieferte Menge verringern, alten Rahmen evtl. wieder öffnen.
    IF    (( old.ag_rahmen_ag_id IS NOT null ) AND ( new.ag_rahmen_ag_id IS DISTINCT FROM old.ag_rahmen_ag_id ))    -- Rahmenbezug wird geändert
        OR ( new.ag_storno AND NOT old.ag_storno )                                                                  -- oder Abruf wird storniert
    THEN
        UPDATE auftg SET
          ag_done = tauftg.auftgr__rahmen_info__stko_offen__by__ag_id__get( old.ag_rahmen_ag_id ) <= 0,
          ag_stkl = tauftg.auftgr__rahmen_info__stkl__by__ag_id__get( old.ag_rahmen_ag_id ) 
          -- Menge geliefert um alten Betrag verringern, da der Rahmen keinen Bezug mehr zur Liefermenge des entfernten Abrufs hat.
        WHERE ag_id = old.ag_rahmen_ag_id;
    END IF;

    -- Mengenänderungen bei vorhandenem Rahmenbezug bzw. Rahmenbezug wird nachgesetzt
    IF (new.ag_rahmen_ag_id IS NOT NULL) AND (NOT new.ag_storno) THEN
        IF (old.ag_stk_uf1 <> new.ag_stk_uf1) OR (old.ag_rahmen_ag_id IS NULL) THEN -- Abrufmenge ändert sich oder neuer Rahmenbezug, Rahmen schließen wenn erfüllt
            UPDATE auftg SET
                ag_done=true
            WHERE ag_id=new.ag_rahmen_ag_id
              AND NOT ag_done
              AND tauftg.auftgr__rahmen_info__stko_offen__by__ag_id__get(new.ag_rahmen_ag_id)<=0;
        END IF;

        IF ( old.ag_stkl <> new.ag_stkl ) OR ( new.ag_rahmen_ag_id IS DISTINCT FROM old.ag_rahmen_ag_id ) THEN 
        -- gelieferte Menge ändert sich oder neuer Rahmenbezug, Rahmen schließen wenn voll be- oder überliefert, gelieferte Menge im Rahmen setzen
            UPDATE auftg SET
                ag_done = ( ag_stk_uf1<=tauftg.auftgr__rahmen_info__stkl__by__ag_id__get( new.ag_rahmen_ag_id )) OR ag_done,
                ag_stkl = tauftg.auftgr__rahmen_info__stkl__by__ag_id__get( new.ag_rahmen_ag_id )
            WHERE ag_id = new.ag_rahmen_ag_id;
        END IF;
    END IF;

    PERFORM enablemodified();
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_u__rahmenauftgupdate
    AFTER UPDATE OF ag_stkl, ag_stk, ag_mcv, ag_stk_uf1, ag_rahmen_ag_id, ag_storno
    ON auftg
    FOR EACH ROW
    WHEN (COALESCE(new.ag_rahmen_ag_id, old.ag_rahmen_ag_id) IS NOT NULL) --nur für Abrufe
    EXECUTE PROCEDURE auftg__a_u__rahmenauftgupdate();
 --

 --Menge ändern bei einer laufenden Position=Bedarfsberechnung aktualisieren
 --Durch Eintrag in art wird die Bedarfsberechnung angestossen
 CREATE OR REPLACE FUNCTION auftg__a_u__stk_bedarfupdate() RETURNS TRIGGER AS $$
  DECLARE beistell BOOL;
  BEGIN
    SELECT agmi_beistell INTO beistell FROM auftgmatinfo WHERE agmi_ag_id=new.ag_id;
    IF beistell IS NULL THEN
        beistell:=FALSE;
    END IF;
    --
    /*wenn sich was an Artikel / Menge ändert, Art nachziehen*/
    IF NOT beistell THEN
        PERFORM tartikel.prepare_artikel_bedarf(old.ag_aknr, true);--2 mal gleiche artikelnummer wird intern abgefangen
        PERFORM tartikel.prepare_artikel_bedarf(new.ag_aknr, true);
    END IF;--NOT beistell
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

   CREATE TRIGGER auftg__a_u__stk_bedarfupdate
    AFTER UPDATE
    OF ag_astat, ag_aknr, ag_stk, ag_stkl, ag_mcv, ag_nbedarf,
       -- Zuschnittsänderungen führen zu Mengenänderung und somit zu bedarfsanpassung #14720
       ag_o6_stkz, ag_o6_lz, ag_o6_zz, ag_o6_hz, ag_o6_bz, ag_o6_zme
       --ag_done => eigener Trigger
    ON auftg
    FOR EACH ROW
    WHEN (NOT new.ag_done)
    EXECUTE PROCEDURE auftg__a_u__stk_bedarfupdate();
 --

 --schliessen einer Position, weitergabe an Art
 CREATE OR REPLACE FUNCTION auftg__a_u__agdone_close() RETURNS TRIGGER AS $$
  DECLARE beistell BOOL;
  BEGIN
   --Set-ABK: Wenn unter ABK nur Material und jetzt alles Material erledigt, dann ist auch ABK erledigt
    IF new.ag_parentabk IS NOT NULL THEN
        IF NOT ab_done FROM abk WHERE ab_ix=new.ag_parentabk THEN --ParentABK ist offen, Set?
            IF NOT EXISTS(SELECT true FROM ab2 WHERE a2_ab_ix=new.ag_parentabk) THEN --keine AG, nur Material > Set
               IF NOT EXISTS(SELECT true FROM auftg WHERE NOT ag_done AND ag_id<>new.ag_id AND ag_parentabk=new.ag_parentabk) THEN --Keine weitere offene Materialposition dieser ABK
                  UPDATE abk SET ab_done=TRUE WHERE ab_ix=new.ag_parentabk;
               END IF;
            END IF;
        END IF;
    END IF;
   --NUN NUR NOCH BEDARFSZEUG, WENN NÖTIG! BEACHTE RETURNS
   IF (new.ag_stk_uf1-new.ag_stkl=0) THEN
        RETURN new;
   END IF;
   --Beistellung
    SELECT agmi_beistell INTO beistell FROM auftgmatinfo WHERE agmi_ag_id=new.ag_id;
    IF beistell IS NULL THEN
        beistell:=FALSE;
    END IF;
    --
    IF beistell THEN--Beistellmaterial ist nicht in der Bedarfsberechnung!
        RETURN new;
    END IF;
   --
    PERFORM tartikel.prepare_artikel_bedarf(old.ag_aknr, true);
    PERFORM tartikel.prepare_artikel_bedarf(new.ag_aknr, true);
   --ACHTUNG, Return bei Beistellung weiter oben!
   RETURN new;
  END $$ LANGUAGE plpgsql;

   CREATE TRIGGER auftg__a_u__agdone_close
    AFTER UPDATE
    OF ag_done, ag_storno
    ON auftg
    FOR EACH ROW
    WHEN ((NOT old.ag_done AND new.ag_done AND (new.ag_pos>0 OR twawi.auftg__ag_pos0defini())) AND (new.ag_astat<>'A'))
    EXECUTE PROCEDURE auftg__a_u__agdone_close();
 --
 --Geschlossene Position wieder öffnen
 --Durch Eintrag in art wird die Bedarfsberechnung angestossen
 CREATE OR REPLACE FUNCTION auftg__a_u__agdone_reopen() RETURNS TRIGGER AS $$
  DECLARE res NUMERIC;
          beistell BOOL;
  BEGIN
   --Set-ABK: Wenn unter ABK nur Material und jetzt alles Material erledigt, dann ist auch ABK erledigt
    IF new.ag_parentabk IS NOT NULL THEN
        IF ab_done FROM abk WHERE ab_ix=new.ag_parentabk THEN --ParentABK ist geschlossen, Set?
            IF NOT EXISTS(SELECT true FROM ab2 WHERE a2_ab_ix=new.ag_parentabk) THEN --keine AG, nur Material > Set
               UPDATE abk SET ab_done=FALSE WHERE ab_ix=new.ag_parentabk;
            END IF;
        END IF;
    END IF;
   --
   --NUN NUR NOCH BEDARFSZEUG, WENN NÖTIG! BEACHTE RETURNS
   IF (new.ag_stk_uf1-new.ag_stkl=0) THEN
        RETURN new;
   END IF;
   --Beistellung
    SELECT agmi_beistell INTO beistell FROM auftgmatinfo WHERE agmi_ag_id=new.ag_id;
    IF beistell IS NULL THEN
        beistell:=FALSE;
    END IF;
    --
    IF beistell THEN--Beistellmaterial ist nicht in der Bedarfsberechnung!
        RETURN new;
    END IF;
    --
    res:=new.ag_stk_uf1-new.ag_stkl;
    IF res<0 THEN
        res:=0;
    END IF;
   --
    PERFORM tartikel.prepare_artikel_bedarf(old.ag_aknr, true);
    PERFORM tartikel.prepare_artikel_bedarf(new.ag_aknr, true);
   --
    RETURN new;
  END $$ LANGUAGE plpgsql;

   CREATE TRIGGER auftg__a_u__agdone_reopen
    AFTER UPDATE
    OF ag_done --kein Storno, Storno entfernen öffnet nicht automatisch ag_done!
    ON auftg
    FOR EACH ROW
    WHEN ((old.ag_done AND NOT new.ag_done AND (new.ag_pos>0 OR twawi.auftg__ag_pos0defini())) AND (new.ag_astat<>'A'))
    EXECUTE PROCEDURE auftg__a_u__agdone_reopen();

 --bei terminschiebung, diesen an die unterpositionen weitergeben
 --bedarfsaktualisierung im zeitverlauf anstoßen
 CREATE OR REPLACE FUNCTION auftg__a_u__kldatum_bedarf() RETURNS TRIGGER AS $$
  BEGIN
    IF     (new.ag_astat <> 'I')
       AND (   new.ag_ldatum IS DISTINCT FROM old.ag_ldatum 
            OR new.ag_kdatum IS DISTINCT FROM old.ag_kdatum 
            )
    THEN -- Haupt / Unterposition (gibts bei Intern nicht!)
        PERFORM disableauftgtrigger();
        --Unterpositionen immer auf gleichem Datum halten
        UPDATE auftg 
           SET ag_ldatum = new.ag_ldatum, 
               ag_kdatum = new.ag_kdatum 
         WHERE ag_astat = new.ag_astat 
           AND ag_nr = new.ag_nr 
           AND ag_hpos = new.ag_pos 
           AND NOT ag_done 
           AND (new.ag_ldatum IS DISTINCT FROM ag_ldatum OR new.ag_kdatum IS DISTINCT FROM ag_kdatum);
        PERFORM enableauftgtrigger();
    END IF;
    --bedarfsaktualisierung
    IF        (new.ag_astat NOT IN ('A'))
      AND NOT (new.ag_done)
    THEN
        PERFORM tartikel.prepare_artikel_bedarf(old.ag_aknr);
        PERFORM tartikel.prepare_artikel_bedarf(new.ag_aknr);
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;
  
   CREATE TRIGGER auftg__a_u__kldatum_bedarf
    AFTER UPDATE
    OF ag_kdatum, ag_ldatum, ag_aldatum-- ag_termv > ist im Einkauf das zuerst bestätigte Lieferdatum
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_u__kldatum_bedarf();
 --
 --Hauptposition beenden, stornieren, oder wieder öffnen auf unterpositionen übertragen

 CREATE OR REPLACE FUNCTION auftg__a_u_donestornosubpos() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM disablemodified();
    IF new.ag_done AND NOT old.ag_done OR new.ag_storno AND NOT old.ag_storno THEN
        UPDATE auftg SET ag_done=new.ag_done OR ag_stk_uf1=ag_stkl OR ag_stk_uf1=ag_stkf, ag_storno=new.ag_storno WHERE ag_id IN (SELECT * FROM auftg_do_auftgsubpos(new.ag_id, true)) AND (ag_done IS DISTINCT FROM new.ag_done OR ag_storno IS DISTINCT FROM new.ag_storno);--alle unterpos schliessen / stornieren
    END IF;
    PERFORM enablemodified();
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_u_donestornosubpos
    AFTER UPDATE
    OF ag_done, ag_storno
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_vkptotalpos)
    EXECUTE PROCEDURE auftg__a_u_donestornosubpos();
 --
--

--Preis enthält Unterposition !!!!!!!!!!!!!!!!!! BEACHTE   FUNCTION auftg__b_iu__possum() RETURNS TRIGGER AS $$
 --In der Hauptposition ändert sich ein Umsatzwert, dann die Unterposition neu berechnen und den Preis der Hauptposition neu auf die Unterpositionen verteilen
 --falls wir Selbst eine Unterposition einer "Preis enthält Unterposition"-Hauptposition sind, dann müssen wir unsere eigene Änderung wiederrum nach oben geben
 CREATE OR REPLACE FUNCTION auftg__a_u__totalposwert_hauptsubpos() RETURNS TRIGGER AS $$
  DECLARE
    vkpupos NUMERIC(16,4);
    anzupos NUMERIC(16,4);
    rundabweichung NUMERIC(16,4);
    mainagid INTEGER;
    grab NUMERIC;
    pwCalc NUMERIC(16,4);
    self_is_subtotalpos_with_toppos BOOL;
  BEGIN
    --
    IF new.ag_vkptotalpos AND NOT old.ag_vkptotalpos THEN--Position wird neu zu totalpos > wir geben unseren Wert nicht nach unten (0) sondern nehmen den Wert von Unten an
        PERFORM tauftg.recalcmainposwert( new.ag_id );
        RETURN new;
    END IF;
    --

    IF auftgtriggerdisabled_totalposwertsubpos() THEN --wir werden durch eine Unterposition aktualisiert, dann geben wir uns selbst nicht wieder nach unten
        --Rekursion nach oben

        IF new.ag_hpos IS NOT NULL THEN
            PERFORM tauftg.recalcmainposwert( auftg_get_mainpos( new, true ) );
        END IF;

        RETURN new;

    END IF;

    --
    PERFORM disableauftgtrigger();
    PERFORM disableauftgtrigger_subpostotalposwert();

    --Gesamtrabatt Hauptposition
    grab   := atd_gesrab FROM auftgdokutxt WHERE atd_dokunr = new.ag_dokunr;
    grab   := ( 1 - COALESCE(grab, 0) / 100);

    self_is_subtotalpos_with_toppos := False;

    -- wir sind eine vkptotal-Unterposition
    IF auftg_get_mainpos( new, true, true) IS NOT NULL THEN

       -- ich selbst bin in einer anderen Position enthalten. Daher bin ich durch die obere Position bereits
       -- gesamtrabattiert und muß entsprechend die Unterpositionen auch entgegengesetzt nehmen
       self_is_subtotalpos_with_toppos := True;

       grab := 1;

    END IF;

    -- Wert aktueller Position
    -- enthält bereits gesamtrabatt, daher aus Unterpositionen herausrechnen
    pwCalc := auftg_pos_wert_calc( new.ag_id, false, false, true, new.ag_nstatistik, new.ag_storno, NOT self_is_subtotalpos_with_toppos) / grab;

    -- Wert aller umsatzwirksamen Unterpositionen, ACHTUNG: ohne Gesamtrabatt. Wichtig, wenn Positionen teilweise im gleichen Dokument wie Hauptposition
    SELECT
       SUM(
          auftg_pos_wert_calc(ag_id, false, false, true, new.ag_nstatistik, new.ag_storno, False)
       ),
       COUNT(1)
    INTO vkpupos, anzupos
    FROM auftg
    WHERE
          ag_id IN ( SELECT * FROM auftg_do_auftgsubpos(new.ag_id, True) )
      AND ( NOT ag_nstatistik OR new.ag_nstatistik)
      AND ag_storno = new.ag_storno;

    -- Wenn Unterpos. Preise haben, Hauptpositionspreis wert-anteilig auf Unterpositionen aufteilen und da reinschreiben
    IF COALESCE( vkpupos, 0 ) > 0 THEN


        UPDATE auftg
        SET
          ag_vkp = pwCalc / Do1If0(vkpupos) * ag_vkp,
          -- Preiseinheit mitnehmen
          ag_preis = (pwCalc / Do1If0(vkpupos) * ag_vkp) * ag_preiseinheit
        WHERE
              ag_id IN (SELECT * FROM auftg_do_auftgsubpos( new.ag_id, True ) )
          AND ( NOT ag_nstatistik OR new.ag_nstatistik)
          AND ag_storno = new.ag_storno;

    ELSE
        -- Wenn Unterpos. keinen Preis haben, Hauptpositionspreis nach Anzahl der Unterpostionen aufteilen und da reinschreiben
        UPDATE auftg
        SET
            ag_vkp = pwCalc / Do1If0(anzupos),
            ag_preis = (pwCalc / Do1If0(anzupos)) * ag_preiseinheit
        WHERE
              ag_id IN ( SELECT * FROM auftg_do_auftgsubpos( new.ag_id, True ) )
          AND ( NOT ag_nstatistik OR new.ag_nstatistik)
          AND ag_storno = new.ag_storno;

    END IF;

    -- Neu durchrechen nach Aktualisierung
    SELECT
      SUM(
          auftg_pos_wert_calc(ag_id, false, false, true, new.ag_nstatistik, new.ag_storno, False)
      )
    INTO vkpupos
    FROM auftg
    WHERE
          ag_id IN (SELECT * FROM auftg_do_auftgsubpos(new.ag_id, True) )
      AND ( NOT ag_nstatistik OR new.ag_nstatistik )
      AND ag_storno = new.ag_storno;

    -- Wieder Hauptpos. - Summe(aller Unterpos) wegen Rundungsfehlern
    rundabweichung := pwCalc - vkpupos;

    IF rundabweichung <> 0 THEN

        UPDATE auftg
        SET
          ag_vkp   = ag_vkp + rundabweichung / Do1If0(ag_stk),
          ag_preis = (ag_vkp+rundabweichung / Do1If0(ag_stk)) * ag_preiseinheit
        WHERE
              ag_id = (
                  SELECT ag_id FROM auftg_do_auftgsubpos(new.ag_id, True)
                  JOIN auftg ON ag_id = auftg_do_auftgsubpos
                  WHERE
                        ag_stk > 0 AND ( NOT ag_nstatistik OR new.ag_nstatistik )
                    AND ag_storno = new.ag_storno
                  ORDER BY ag_stk
                  LIMIT 1
              );
    END IF;
    --
    PERFORM enableauftgtrigger_subpostotalposwert();
    PERFORM enableauftgtrigger();

    RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_u__totalposwert_hauptsubpos
    AFTER UPDATE
    OF ag_stk, ag_vkp, ag_arab, ag_netto, ag_kurs, ag_canrabatt, ag_vkptotalpos, ag_preis, ag_preiseinheit
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_vkptotalpos)
    EXECUTE PROCEDURE auftg__a_u__totalposwert_hauptsubpos();
 --
 --Wert der Hauptposition aktualisieren, dazu auf die Hauptposition gehen und von dort aus den Wert aller Werthaltigen Unterpositionen kalkulieren, anschliessend diesen zurückschreiben in die Hauptposition
 CREATE OR REPLACE FUNCTION tauftg.recalcmainposwert(
     IN mainagid integer,
     -- Falls die Position *NICHT* Preis enthält Unterposition hat, dennoch den Wert so berechnen, als hätte sie das flag
     -- somit wird der Preis der angegebenen Position aus den Unterpositionen zusammengerechnet und geschrieben
     IN force_calc boolean = false
     )
     RETURNS numeric(16,4)
     AS $$
     DECLARE vkpupos numeric(16,4);
             anzupos numeric(16,4);
             abzunetto numeric(16,4);
             vkptotalpos boolean;
             mainposisopt boolean;
             stornopos boolean;
             result numeric(16,4);
     BEGIN

       -- Position, welche angegeben ist - ist *KEINE* Preis enthält Unterposition
       -- dann gehen wir im Standard raus. Bei FroceCalc berechnen wir dennoch die Position neu!
       IF     NOT force_calc
          AND NOT ag_vkptotalpos FROM auftg WHERE ag_id = mainagid
       THEN
          RETURN null;
       END IF;

       IF    coalesce(mainagid, 0) = 0
          OR auftgtriggerdisabled_subpostotalposwert()
       THEN
          RETURN null;
       END IF;
       --
       PERFORM disableauftgtrigger();
       PERFORM disableauftgtrigger_totalposwertsubpos();
       --
       SELECT ag_nstatistik, ag_storno
         INTO mainposisopt, stornopos
         FROM auftg
        WHERE ag_id = mainagid;

       -- VKP-Summe und Anzahl der Unterpositionen
       SELECT sum( auftg_pos_wert_calc(ag_id, false, false, true, mainposisopt, stornopos) ),
              count(1)
         INTO vkpupos, anzupos
         FROM auftg
        WHERE ag_id IN ( SELECT * FROM auftg_do_auftgsubpos(mainagid, true) )
          AND ag_nstatistik = mainposisopt
          AND ag_storno = stornopos;

       -- Abzuschläge der hauptposition abziehen
       abzunetto := twawi.auftg__abzu__netto__by__ag_id(mainagid);
       vkpupos   := vkpupos - coalesce(abzunetto, 0);
       --
       --
       UPDATE auftg
          SET ag_vkp   = (  coalesce(vkpupos, 0)
                          / IFTHEN(ag_canrabatt,
                                   (1 - ag_arab / 100),
                                   1
                                   )
                          / ag_stk
                          ),
              ag_preis = (  coalesce(vkpupos, 0)
                          / IFTHEN(ag_canrabatt,
                                   (1 - ag_arab / 100),
                                   1
                                  )
                          / ag_stk
                          )
                        * ag_preiseinheit
        WHERE ag_id = mainagid
          AND ag_stk > 0
        RETURNING ag_vkp
             INTO result;
       --
       PERFORM enableauftgtrigger_totalposwertsubpos();
       PERFORM enableauftgtrigger();
       --
       RETURN result;
     END $$ LANGUAGE plpgsql;
 --

 --Eine Unterposition wird aus der Hauptposition herausgezogen, dann die Hauptposition neu berechnen (der Unterpositionswert ist ja jetzt raus)
 CREATE OR REPLACE FUNCTION auftg__a_iu__subposwertchangemainpos() RETURNS TRIGGER AS $$
  DECLARE
    _auftg_mainpos INTEGER;
  BEGIN
    --Eine Unterposition wird aus einer Hauptposition herausgezogen, Hauptposition neu berechnen
    IF tg_op = 'UPDATE' THEN
        IF ( old.ag_hpos IS NOT NULL ) AND ( new.ag_hpos IS DISTINCT FROM old.ag_hpos ) THEN

            BEGIN
              _auftg_mainpos := auftg_get_mainpos( old, true );
            EXCEPTION

              -- im normalfall wird eine unterposition von einer position »a« nach »b« umgezogen
              --     a   b             a   b
              --      \      hin zu:      /
              --       c                 c
              -- für diesen fall ist die verwendung von old korrekt.
              -- wenn aber für »a« die ag_pos geändert wird, wird die via update cascade für position »c« die ag_hpos durchgereicht.
              -- durch die triggerverwendung entsteht ein inkonsistenter datenzustand 'zwischen' den triggerphasen
              -- daher die sonderbehandlung an der stelle, weil in diesem kontext die mainpos für auftg auf new gehen müsste,
              -- was wiederrum hier nicht zu erkennen ist, ob ich auf new oder old muss. daher schlägt die selectierung des korrekten parent fehl
              -- was aber nicht weiter relevant ist, da die positionen gemeinsam umgesetzt werden und es keiner neuberechnung bedarf.

              WHEN SQLSTATE '23503' THEN
                _auftg_mainpos := null;
            END;

            --wenn wir schieben sich aber in der Hauptpostion nichts ändert, wir ändern zB nur eine Reihenfolge
            IF _auftg_mainpos = auftg_get_mainpos( new, true ) THEN
                RETURN new;
            END IF;

            PERFORM tauftg.recalcmainposwert( _auftg_mainpos );
        END IF;

    END IF;

    -- Eine Unterposition wird in eine Hauptposition hineingezogen, oder sie ändert ihren Wert oder wird storniert
    IF new.ag_hpos IS NOT NULL THEN
        PERFORM tauftg.recalcmainposwert( auftg_get_mainpos( new, true ) );
    END IF;

    RETURN new;

  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu__subposwertchangemainpos
    -- muß nach Cache aktualisieren sein, da sonst noch alte Werte gesehen werden
    -- beachte bei Änderungen: belzeil__a_u__totalposwert_hauptsubpos
    AFTER INSERT OR UPDATE
    OF ag_hpos, ag_stk, ag_vkp, ag_arab, ag_netto, ag_kurs, ag_canrabatt, ag_nstatistik, ag_storno, ag_preis, ag_preiseinheit
    ON auftg
    FOR EACH ROW
    --WHEN (NOT new.ag_vkptotalpos) NICHT, da ich eine "Preis enthält Unterposition" - Position in eine andere schieben kann, wodurch sich diese ändern muß
    EXECUTE PROCEDURE auftg__a_iu__subposwertchangemainpos();

  CREATE OR REPLACE FUNCTION auftg__a_d__ag_hpos__subposwertchangemainpos() RETURNS TRIGGER AS $$
   DECLARE
     _mainagid  INTEGER;
     _parent_auftg_record RECORD;
   BEGIN

     PERFORM tauftg.recalcmainposwert( auftg_get_mainpos( old, true ) );

     RETURN old;

   END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_d__ag_hpos__subposwertchangemainpos
    AFTER DELETE
    ON auftg
    FOR EACH ROW
    WHEN (old.ag_hpos IS NOT NULL)
    EXECUTE PROCEDURE auftg__a_d__ag_hpos__subposwertchangemainpos();

-- ENDE Preis enthält Unterposition
 CREATE OR REPLACE FUNCTION auftg__a_u__ag_dispokrzl() RETURNS TRIGGER AS $$
  BEGIN
    UPDATE auftgdokutxt SET atd_apkrzl = new.ag_dispokrzl, atd_ap = new.ag_dispo  WHERE atd_dokunr = new.ag_dokunr AND atd_apkrzl IS NULL AND atd_ap IS NULL;
    RETURN new;
  END $$ LANGUAGE plpgsql;

  --
  CREATE TRIGGER auftg__a_u__ag_dispokrzl
   AFTER UPDATE OF ag_dispokrzl, ag_dispo
   ON auftg
   FOR EACH ROW
   WHEN (new.ag_dokunr IS NOT NULL)
   EXECUTE PROCEDURE auftg__a_u__ag_dispokrzl();
 --
 CREATE OR REPLACE FUNCTION auftg__a_iu__aknrXXXX() RETURNS TRIGGER AS $$
  BEGIN
    IF (new.ag_parentabk=new.ag_ownabk) THEN
        RAISE EXCEPTION 'xtt4909 %', Format(lang_text(29172) /*Eine ABK kann sich nicht selbst enthalten: eigene ABK:%, parent ABK:%'*/, new.ag_post3, new.ag_post4);
    END IF;
    IF tg_op='INSERT' THEN  --hinweis: durch die OF Spalten ist hier klar, das es immer NK-Relevant ist früher gab es eine Feldprüfung wo jedes relevante Feld verglichen wurde
        UPDATE abk SET ab_nk_change=TRUE WHERE ab_ix=new.ag_parentabk AND NOT ab_nk_change; --AND ag_aknr<>ag_aknr OR ag_preis<>preis OR usw
    END IF;
    --
    IF tg_op='UPDATE' THEN --new.ag_aknr<>old.ag_aknr im Update direkt
        UPDATE abk SET ab_ap_nr=new.ag_aknr WHERE ab_ix=new.ag_ownabk AND ab_ap_nr=old.ag_aknr AND new.ag_aknr<>old.ag_aknr;
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu__aknrXXXX
    AFTER INSERT OR UPDATE
    OF ag_aknr, ag_stk, ag_vkp, ag_stkl, ag_nk_vkp, ag_nk_calc_vkp_uf1, ag_nk_stkl, ag_preis, ag_preiseinheit
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_iu__aknrXXXX();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_iu__projupdate() RETURNS TRIGGER AS $$
  BEGIN
    -- Trigger-Bedingung: Projektnummer in Auftrag gesetzt und keine ABK-Materialpositon
      -- WHEN ( new.ag_an_nr IS NOT NULL AND new.ag_astat <> 'I' )

    -- Projekt-Adressen werden automatisch vom 1. zugehörigen Auftrag bzw. Angebot gefüllt.
      -- wenn globales Setting dies erlaubt (default true).
    IF TSystem.Settings__GetBool( 'anl_addresses__set__by__auftg', true ) THEN

        -- Standort
        UPDATE anl SET
          an_sortkrz = new.ag_krzl
        WHERE an_sortkrz IS NULL
          AND an_nr = new.ag_an_nr
          AND new.ag_krzl <> TAdk.get_vorg_ladress('#')
        ;

        -- Besteller
        UPDATE anl SET
          an_best = new.ag_lkn
        WHERE an_best IS NULL
          AND an_nr = new.ag_an_nr
          AND new.ag_lkn <> '#'
        ;

        -- Kunde und Betreiber
        UPDATE anl SET
          an_kund = new.ag_lkn,
          an_hest = new.ag_lkn
        WHERE an_kund IS NULL
          AND an_nr = new.ag_an_nr
          AND new.ag_lkn <> '#'
        ;

    END IF;


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu__projupdate
    AFTER INSERT OR UPDATE
    OF ag_krzl, ag_lkn, ag_an_nr
    ON auftg
    FOR EACH ROW
    WHEN ( new.ag_an_nr IS NOT NULL AND new.ag_astat <> 'I' )
    EXECUTE PROCEDURE auftg__a_iu__projupdate();
 --

 --
 -- AngebotsNummer in Kundenanfragen Anfragepositionen schreiben
 CREATE OR REPLACE FUNCTION auftg__a_id() RETURNS TRIGGER AS $$
  BEGIN
    IF TG_OP='INSERT' THEN
        IF new.ag_kanf_nr IS NOT NULL THEN
            UPDATE kundanfragepos SET
              kanfp_angnr= new.ag_nr,
              kanfp_ag_id= new.ag_id
            WHERE kanfp_kanf_nr = new.ag_kanf_nr
              AND kanfp_ak_nr = new.ag_aknr
              AND kanfp_angnr IS NULL;  -- abfangen der einträge per hand
        END IF;
        IF new.ag_bda IS NULL THEN -- Kundenreferenznummer übertragen
            UPDATE auftg SET ag_bda=kanf_refnummer FROM kundanfrage WHERE ag_nr = new.ag_nr AND kanf_nr = new.ag_kanf_nr;
        END IF;
        IF new.ag_kanfwsk_wsk IS NULL THEN -- Realisierungswarscheinligkeit übertragen   #6679
            UPDATE auftg SET ag_kanfwsk_wsk = kanf_kanfwsk_wsk FROM kundanfrage WHERE ag_nr = new.ag_nr AND kanf_nr = new.ag_kanf_nr;
        END IF;

        RETURN new;
    ELSE --DELETE
        IF old.ag_kanf_nr IS NOT NULL THEN
            UPDATE kundanfragepos SET
              kanfp_angnr= NULL
            WHERE kanfp_kanf_nr = old.ag_kanf_nr
              AND kanfp_ak_nr = old.ag_aknr
              AND kanfp_angnr = old.ag_nr -- abfangen der einträge per hand
              AND kanfp_ag_id IS NULL; -- aufgrund Referenz mit ON DELETE SET NULL
        END IF;

        RETURN old;
    END IF;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_id
    AFTER INSERT OR DELETE
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_id();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_d() RETURNS TRIGGER AS $$
  DECLARE res NUMERIC;
          mainagid INTEGER;
  BEGIN
    IF old.ag_dokunr IS NOT NULL THEN
        RAISE EXCEPTION 'xtt11804 - auftg delete dokunr exist %', old.ag_dokunr;
    END IF;
    --

    -- Falls Unterpositionen vorhanden, diese löschen
    UPDATE auftg
       SET ag_dokunr = NULL
     WHERE ag_dokunr IS NOT NULL
       AND ag_hpos  = old.ag_pos
       AND ag_astat = old.ag_astat
       AND ag_nr    = old.ag_nr;

    -- Falls Unterpositionen vorhanden, diese löschen
    DELETE FROM auftg
     WHERE ag_hpos  = old.ag_pos
       AND ag_astat = old.ag_astat
       AND ag_nr    = old.ag_nr;

    -- Angebot wieder öffnen, außer es gibt noch weitere Aufträge, dia auch Zuordnung zum gleichen Angebot haben
    UPDATE auftg SET ag_done = FALSE
     WHERE ag_id = old.ag_ang_ag_id AND NOT EXISTS (SELECT true FROM auftg WHERE ag_ang_ag_id=old.ag_ang_ag_id);
    --
    -- Hauptposition aktualisieren, welche Preise der Unterpos enthält
    -- EXISTS: Beim Löschen eines kompletten Astes die Mehrfachausführung verhindern, die durch DELETE aller Unterpos ausgelöst werden würde (Trigger-Rekursion)
    -- (im AFTER DELETE kennen die Unterpos den gelöschten Ast nicht mehr, also nur ausführen wenn Ast noch existiert)
    IF     (old.ag_hpos IS NOT NULL)
       AND EXISTS(SELECT true FROM auftg
                   WHERE ag_astat = old.ag_astat
                     AND ag_nr    = old.ag_nr
                     AND ag_pos   = old.ag_hpos
                  )
    THEN
        -- erster gültiger Ast
        mainagid := (SELECT ag_id FROM auftg
                      WHERE ag_astat = old.ag_astat
                        AND ag_nr    = old.ag_nr
                        AND ag_pos   = old.ag_hpos);

        -- rekursive Ermittlung der zu aktualisierenden Hpos (aufgrund Preis enthält Unterpos)
        mainagid := coalesce(auftg_get_auftgmainpos(mainagid, true), mainagid);

        IF ag_vkptotalpos FROM auftg WHERE ag_id = mainagid THEN
           PERFORM tauftg.recalcmainposwert(mainagid);
        END IF;
    END IF;
    --
    -- Wenn Rahmenabruf gelöscht wird, dann gelieferte Menge verringern, alten Rahmen evtl. wieder öffnen.
    IF         (old.ag_rahmen_ag_id IS NOT NULL)
       AND (NOT old.ag_storno)
    THEN
        PERFORM disablemodified();

        UPDATE auftg
           SET ag_done = tauftg.auftgr__rahmen_info__stko_offen__by__ag_id__get(old.ag_rahmen_ag_id) <= 0,
               ag_stkl = ag_stkl - old.ag_stkl
         WHERE ag_id = old.ag_rahmen_ag_id;

        PERFORM enablemodified();
    END IF;

    -- Artikel-Bedarf
    IF     (   old.ag_pos > 0
            OR twawi.auftg__ag_pos0defini()
            )
       AND (    old.ag_astat <> 'A')
       AND (NOT old.ag_done)
    THEN
       PERFORM tartikel.prepare_artikel_bedarf(old.ag_aknr, true);
    END IF;

    -- Kopftabellem zum Auftrag! Prüfen ob es noch Positionen gibt und ansonsten Kopftabellen löschen
    IF NOT EXISTS(SELECT true FROM auftg WHERE ag_astat = old.ag_astat AND ag_nr = old.ag_nr) THEN
       DELETE FROM auftgtxt WHERE at_astat = old.ag_astat AND at_nr = old.ag_nr;
    END IF;
    IF NOT EXISTS(SELECT true FROM auftg WHERE ag_dokunr = old.ag_dokunr) THEN --Zusatztexte zum Auftrag!
        DELETE FROM auftgdokutxt WHERE atd_dokunr = old.ag_dokunr;
    END IF;
    --

    RETURN old;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_d
    AFTER DELETE
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_d();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__as__a_iud() RETURNS TRIGGER AS $$
  BEGIN
    IF current_user='root' THEN
        RAISE NOTICE 'ENTER FUNCTION auftg__as__a_iu() - %', currenttime();
    END IF;
    --
    PERFORM tcache.function_cache_setdirty_1param('abk_main_auftg_id__extended__fuzzysearch__by__aknr', dab_aknr) FROM do_artikel_bedarf WHERE dab_done IS false;
    PERFORM do_artikel_bedarf();
    --
    IF current_user='root' THEN
        RAISE NOTICE 'FULL EXIT FUNCTION auftg__as__a_iu() - %', currenttime();
    END IF;
    --
    RETURN NULL;
  END $$ LANGUAGE plpgsql;


  CREATE TRIGGER auftg__as__a_iud
    AFTER INSERT OR UPDATE OR DELETE
    --OF ag_astat, ag_done, ag_storno, ag_aknr, ag_mcv, ag_stk, ag_stkl, ag_kdatum, ag_ldatum, ag_twa
    ON auftg
    FOR EACH STATEMENT
    EXECUTE PROCEDURE auftg__as__a_iud();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_iu_ak_kunde() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM disablemodified();
    --
    UPDATE art SET
      ak_kunde = new.ag_lkn
    WHERE ak_nr = new.ag_aknr
      AND ak_kunde IS NULL;
    --
    PERFORM enablemodified();
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu_ak_kunde
    AFTER INSERT OR UPDATE OF ag_lkn, ag_aknr
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_astat IN ('E', 'A'))
    EXECUTE PROCEDURE auftg__a_iu_ak_kunde();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_u_ag_parentabk() RETURNS TRIGGER AS $$
  BEGIN
    -- Warenausgänge mit umziehen, wenn Material verschoben wird. Wegen Rückbuchung, Kommissionierung.
    UPDATE lifsch 
       SET l_ab_ix = new.ag_parentabk
     WHERE l_ab_ix IS DISTINCT FROM new.ag_parentabk 
       AND l_ag_id = new.ag_id 
       AND l_ab_ix = old.ag_parentabk;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_u_ag_parentabk
    AFTER UPDATE OF ag_parentabk
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_u_ag_parentabk();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_iu__doprognose() RETURNS TRIGGER AS $$
  BEGIN
    UPDATE bestanfpos 
       SET bap_done = new.ag_done 
     WHERE bap_done IS DISTINCT FROM new.ag_done 
       AND bap_pr_ag_id = new.ag_id;--bestehende Positionen schliessen/wieder öffnen!
    --
    IF (NOT new.ag_done) AND TSystem.Settings__GetBool('ArtikelBedarfPrognose') AND tartikel.has_vorproduktion_stv(new.ag_aknr) THEN --Position läuft und es existieren Unterartikel
        --achtung: bei Änderungen muß DailyDBFunctions.PrognosenAktualisieren angepasst werden!
        PERFORM tplanterm.artikel_prognose_from_auftg(new);
    END IF;
    --
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu__doprognose
    AFTER INSERT OR UPDATE
    OF ag_aknr, ag_stk, ag_stkl, ag_aldatum, ag_ldatum, ag_kdatum, ag_done, ag_storno
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_astat IN ('E', 'A'))--R?
    EXECUTE PROCEDURE auftg__a_iu__doprognose();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_iu__ag_kldatum() RETURNS trigger AS $$
  BEGIN
    -- Weitergabe des Bedarfsdatums der Materialposition an verknüpfte BANF


    UPDATE bestanfpos
       SET bap_termin = coalesce( new.ag_ldatum, new.ag_kdatum )
     WHERE bap_ag_id = new.ag_id
    ;


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu__ag_kldatum
    AFTER INSERT OR UPDATE
    OF ag_kdatum, ag_ldatum
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__a_iu__ag_kldatum();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__b_iu__ag_twa() RETURNS trigger AS $$
  BEGIN
    -- Kalenderwoche formatieren und Format überprüfen.


    IF new.ag_twa IS NOT null THEN
        new.ag_twa := TSystem.termweek__format( new.ag_twa );
    END IF;


    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__ag_twa
    BEFORE INSERT OR UPDATE
    OF ag_twa
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__b_iu__ag_twa();
 --

 -- verschiedene Fehlerbehandlung nach Aufbereitung der Daten
 CREATE OR REPLACE FUNCTION auftg__b_z_iu__error_handling() RETURNS trigger AS $$
  DECLARE old_data RECORD;
  BEGIN
    -- Materialbedarfe für ABKs
    IF new.ag_astat = 'I' THEN
        -- Deadlock verhindern durch zirkuläre ABK-Beziehungen.
        IF TG_OP = 'UPDATE' THEN
            SELECT old.ag_ownabk AS ag_ownabk_old, old.ag_parentabk AS ag_parentabk_old INTO old_data;
        ELSE
            SELECT NULL::INTEGER AS ag_ownabk_old, NULL::INTEGER AS ag_parentabk_old INTO old_data;
        END IF;

        IF new.ag_ownabk IS NOT NULL AND new.ag_parentabk IS NOT NULL -- gibt es überhaupt was zu prüfen (Performance)
            AND (new.ag_ownabk IS DISTINCT FROM old_data.ag_ownabk_old OR new.ag_parentabk IS DISTINCT FROM old_data.ag_parentabk_old) -- gab es überhaupt Änderungen (Performance)
        THEN
            IF new.ag_ownabk IN (SELECT tplanterm.get_all_parent_abk(new.ag_parentabk)) THEN -- eigentliche Prüfung
                RAISE EXCEPTION '%', lang_text(16453) || E'\n\n' || lang_text(105) || ': ' || new.ag_ownabk || E'\n' ||
                    lang_text(929) || ': ' || (SELECT array_to_string(ARRAY(SELECT DISTINCT tplanterm.get_all_parent_abk(new.ag_parentabk)), ' -> '));
            END IF;
        END IF;
        --
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_z_iu__error_handling
    BEFORE INSERT OR UPDATE
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__b_z_iu__error_handling();
  --

  -- Legt beim Insert die 3 Parameter für Durchlaufzeiten automatisch an.
  CREATE TRIGGER auftg__a_i__create_autoparams
    AFTER INSERT
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_astat = 'E')
    EXECUTE PROCEDURE TRecnoParam.CreateAutoParams();
 --

 --
 CREATE OR REPLACE FUNCTION auftg__a_iu_keywordsearch() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM TSystem.kws_create_keywords(new.*);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_iu_keywordsearch
    AFTER INSERT OR UPDATE
     OF ag_nr, ag_an_nr, ag_aknr_referenz, ag_akbz, ag_bda, ag_dokunr, ag_parentabk, ag_ownabk
    ON auftg
    FOR EACH ROW
    --WHEN (new.ag_astat IN ('A', 'E', 'R'))
    EXECUTE PROCEDURE auftg__a_iu_keywordsearch();
 --

 -- #9052 Freigabestatus auf Positionsebene in ABK
 CREATE OR REPLACE FUNCTION auftg__a_75_iu__bstat__sf() RETURNS TRIGGER AS $$
  BEGIN
    -- WHEN (new.ag_astat = 'I' AND NOT new.ag_storno)

    IF TG_OP = 'INSERT' THEN
        IF TSystem.Settings__GetBool('ABK_auftgi__ag_bstat__ag_done__initial_gesperrt') THEN -- Einstellung/Produktionsplanung/ABK-Materialposition initial gesperrt (positionsweise Freigabe)
            UPDATE auftg SET -- Bei Anlage inital alle Materialpositionen sperren.
              ag_done= true,
              ag_bstat= 'S'
            WHERE ag_id = new.ag_id
              AND NOT ag_done;
        END IF;
    ELSE -- UPDATE
        IF NOT (new.ag_done AND new.ag_bstat = 'S') THEN -- INSERT-Fall initial abfangen.
            IF (new.ag_stk_uf1 - new.ag_stkl > 0) AND new.ag_bstat IN ('S', 'F') THEN -- Wenn voll befliefert kein Automatismus.
                IF new.ag_bstat = 'S' AND NOT new.ag_done THEN -- Beim Setzen des Status auf S, Materialposition schließen.
                    UPDATE auftg SET
                      ag_done= true
                    WHERE ag_id = new.ag_id;
                ELSIF old.ag_bstat = 'S' AND new.ag_bstat = 'F' AND new.ag_done THEN -- Beim Setzen des Status von S auf F, Materialposition öffnen/freigeben.
                    UPDATE auftg SET
                      ag_done= false
                    WHERE ag_id = new.ag_id;
                END IF;
            END IF;
        END IF;
    END IF;

    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_75_iu__bstat__sf
    AFTER INSERT OR UPDATE
    OF ag_bstat
    ON auftg
    FOR EACH ROW
    WHEN (new.ag_astat = 'I' AND NOT new.ag_storno)
    EXECUTE PROCEDURE auftg__a_75_iu__bstat__sf();
--

-- automatische Übernahme von auftg
-- #13286 die TV-Kategorie wird nicht mehr automatisch gesetzt
CREATE OR REPLACE FUNCTION public.auftg__a_50_u__terminverschiebung() RETURNS trigger AS $$
  BEGIN
    IF
           old.ag_ldatum IS NOT NULL
       AND new.ag_ldatum IS NOT NULL
       AND old.ag_ldatum <> new.ag_ldatum
    THEN

      INSERT INTO terminverschiebung ( tver_ag_id, tver_termin_alt, tver_termin_neu )
      VALUES                         (  new.ag_id,  old.ag_ldatum,   new.ag_ldatum  );

      PERFORM prodat_hint(
                concat(
                        lang_text( 28959 ), -- Neue Terminverschiebung
                        ': ',
                        old.ag_ldatum,
                        ' -> ',
                        new.ag_ldatum
                      )
              );
    END IF;

    RETURN new;
  END $$  LANGUAGE plpgsql VOLATILE;
  --

  CREATE TRIGGER auftg__a_50_u__terminverschiebung
    AFTER UPDATE
    OF ag_ldatum
    ON auftg
    FOR EACH ROW
    -- Terminverschiebungen nur für externe Aufträge interessant
    WHEN ( new.ag_astat = 'E' )
    EXECUTE PROCEDURE auftg__a_50_u__terminverschiebung();
--

-- Standaort für das Material anhand der Kostenstelle ermitteln
CREATE OR REPLACE FUNCTION auftg__b_iu__ag_a2_id() RETURNS TRIGGER AS $$
    DECLARE _oldks_krz varchar;
    BEGIN
        IF tg_op = 'UPDATE' THEN
           _oldks_krz := ks_krzl FROM ksv JOIN ab2 ON ks_abt = a2_ks WHERE a2_id = old.ag_a2_id;
        END IF;

        -- StandardAdresse, oder alter AG wird umgeschrieben
        IF    Equals(_oldks_krz, new.ag_krzl) -- das alte abweichende war dem alten AG entsprechend
           OR Equals(TAdk.get_vorg_ladress('#'), new.ag_krzl)
        THEN
           new.ag_krzl := coalesce(/*TODO: aus AC: Plantinen immer Hale 3 (LOLL),*/
                                   -- Standort aus Kostenstelle, default interne
                                   (SELECT ks_krzl FROM ksv JOIN ab2 ON ks_abt = a2_ks WHERE a2_id = new.ag_a2_id),
                                   TAdk.get_vorg_ladress('#') -- insert mit keinem AG oder herauslöschen endet hier
                                   );
        END IF;

        RETURN new;
    END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_iu__ag_a2_id
    BEFORE INSERT OR UPDATE
    OF ag_a2_id
    ON auftg
    FOR EACH ROW
    -- Lager UE (Halbfertigteile im Lager) ausschliessen
    WHEN (NOT TSystem.ENUM_GetValue(new.ag_stat, 'UE') )-- ag_nbedarf IS false?
    EXECUTE PROCEDURE auftg__b_iu__ag_a2_id();

--
CREATE OR REPLACE FUNCTION auftg__a_u__ag_a2_id() RETURNS TRIGGER AS $$
    BEGIN
        -- Aktualisierung des Materialtermins bei Änderung des Bezugs zum Arbeitsgang
        -- Beachte Nicht-Ausführen des Triggers über Setting disable__auftg__a_u__ag_a2_id
          -- Anwendungsfall: Direkt nach ABK-Erstellung braucht dieser rechenintensive Prozess nicht durchgeführt werden.
          -- Alternative: Bedingung prüfen: Termin am AG ist nicht gesetzt, braucht also nicht aktualisiert werden.

        PERFORM tabk.abk__auftgi_kdatum__refresh( ARRAY[new.ag_id] );

        RETURN new;
    END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__a_u__ag_a2_id
    AFTER UPDATE
    OF ag_a2_id
    ON auftg
    FOR EACH ROW
    WHEN ( new.ag_astat = 'I' AND NOT TSystem.Settings__GetBool( 'disable__auftg__a_u__ag_a2_id' ) )
    EXECUTE PROCEDURE auftg__a_u__ag_a2_id();
--
--- #17495
CREATE OR REPLACE FUNCTION auftg__b_10_i() RETURNS TRIGGER AS $$
    BEGIN
        IF ( new.ag_astat = 'E' ) AND ( SELECT a1_sperr_auftg FROM adk1 WHERE a1_krz = new.ag_lkn ) THEN
            RAISE EXCEPTION 'xtt29787 a1_sperr_auftg: auftge not allowed';
        END IF;
        RETURN new;
    END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftg__b_10_i
    BEFORE INSERT
    ON auftg
    FOR EACH ROW
    EXECUTE PROCEDURE auftg__b_10_i();
 ---

-- Währung von der Kundenzuordnung übernehmen. siehe Ticket 17473
CREATE OR REPLACE FUNCTION auftg__b_10_i_waer() RETURNS TRIGGER AS $$
  DECLARE
    _az_waco text;
  BEGIN
    /* Wenn vorhanden, die Währung der Kundenzuordnung übernehmen. Es kann mehrere Kundenzuordnungen für eine Kombination aus
        Artikelnummer und Kunde (Bestelladresse) geben. Diese unterscheiden sich dann entweder über den Gültigkeitszeitraum oder
        der Mindestmenge. */
    _az_waco := az_waco
                FROM artzuo
                WHERE     az_prokrz = new.ag_lkn
                      AND az_pronr = new.ag_aknr
                      AND new.ag_bdat BETWEEN coalesce( az_gdatum, '-infinity' ) AND coalesce( az_bisdatum, 'infinity' )
                      AND coalesce( az_minmenge, 0 ) <= new.ag_stk
                ORDER BY az_minmenge DESC
                LIMIT 1;
    -- Wenn vorhanden und noch keine Währung gesetzt, die Währung des Debitor-Datensatzes setzen, ansonsten auf die Standardwährung zurückfallen.
    new.ag_waer := coalesce( _az_waco, adk1__a1_waco__or__basis_w__by__a1_krz__get( new.ag_lkn ) );
    RETURN new;
  END $$ LANGUAGE plpgsql;
  --
  CREATE TRIGGER auftg__b_10_i_waer
    BEFORE INSERT
      ON auftg
    FOR EACH ROW
      WHEN ( new.ag_waer IS NULL )
    EXECUTE PROCEDURE auftg__b_10_i_waer();
--

-- Planauträge (Plantafel Anzeige TMP Data)
 CREATE TABLE auftg_planauftg (
  uteil             BOOLEAN,
  auftgid           TEXT,
  aufdbrid          VARCHAR(100),
  ag_id             INTEGER REFERENCES auftg ON DELETE CASCADE,
  ag_astat          VARCHAR(1),
  ag_nr             VARCHAR(30),
  ag_pos            SMALLINT,
  ag_lkn            VARCHAR(21),
  ak_nr             VARCHAR(40),
  aknr              VARCHAR,
  ak_bez            VARCHAR(100),
  ag_stk            NUMERIC(12,4),
  op_ix             INTEGER,
  term              DATE,
  -- System (tables__generate_missing_fields)
  --   kein automatischer table_delete-Trigger (tables__fieldInfo__fetch)
  dbrid             VARCHAR(32) NOT NULL DEFAULT nextval('db_id_seq')
);



--UMSATZPLANUNG  http://redmine.prodat-sql.de/projects/prodat-v-x/wiki/UmsatzplanungUmsatzrunde
CREATE TABLE auftgliefplanung
     (
      aglp_id               SERIAL PRIMARY KEY,
      aglp_ag_id            INTEGER REFERENCES auftg ON DELETE CASCADE,
      aglp_ident            VARCHAR(50),
      aglp_inumsatz         BOOLEAN NOT NULL DEFAULT false,
      aglp_planyearmonth    VARCHAR(6) NOT NULL,
      aglp_stk              NUMERIC(12,2),
      aglp_manwert          NUMERIC(12,4),
      aglp_log              TEXT
     );

 CREATE INDEX auftgliefplanung_aglp_planyearmonth ON auftgliefplanung (aglp_planyearmonth);

 CREATE OR REPLACE FUNCTION auftgliefplanung__b_u_aglp_inumsatz() RETURNS trigger AS $$
  DECLARE urid_old INTEGER;
          urid_akt INTEGER;  --laufende Umsatzrunde
          countUR   INTEGER;
          PlanedYMnew VARCHAR;
          PlanedYMold VARCHAR;  --das war der Monat, in dem der Umsatz bisher geplant war, wenn er entplant wird
          I   INTEGER;
          S   VARCHAR(100);
   BEGIN
     IF auftgliefplanung__b_u_wertchange_disabled() THEN
        RETURN new;
     END IF;
     --ACHTUNG: umschreiben geht nicht, es muß aus- und eingeplant werden!
     IF new.aglp_inumsatz THEN        --Umsatzrunde Plus, oder Change, merken, wohin wir geplant werden und wo wir geplant waren
        PlanedYMnew:=new.aglp_planyearmonth;
        --Umsatzrunden in betroffenen Monat ermitteln ermitteln
        SELECT MAX(uplr_id), COUNT(1) INTO urid_akt, countUR FROM umsatzplanung_umsatzrunde WHERE uplr_planyearmonth=PlanedYMnew;
        IF COALESCE(countUR,0)=0 THEN
            RAISE EXCEPTION '%', lang_text(29173) /*'keine Umsatzrunde gestartet'*/;
        END IF;
        --
        UPDATE auftg SET ag_bstat1='U' WHERE new.aglp_ag_id=ag_id;
     ELSE               --Umsatzrunde Minus, merken, wo wir waren
        PlanedYMold:=old.aglp_planyearmonth;
        --Umsatzrunden in betroffenen Monat ermitteln ermitteln
        SELECT MAX(uplr_id), COUNT(1) INTO urid_akt, countUR FROM umsatzplanung_umsatzrunde WHERE uplr_planyearmonth=PlanedYMold;
        --
        UPDATE auftg SET ag_bstat1=NULL WHERE new.aglp_ag_id=ag_id;
        new.aglp_manwert:=NULL;
        --
     END IF;
     --Nun ermitteln der ID der Umsatzrunde, in der wir zuletzt bewegt wurden
     urid_old:=uplrpm_uplr_id FROM umsatzplanung_umsatzrunde JOIN umsatzplanung_umsatzrunde_plusminus ON uplr_id=uplrpm_uplr_id WHERE uplr_planyearmonth=PlanedYMold AND uplrpm_aglp_id=new.aglp_id ORDER BY uplr_id DESC LIMIT 1;--da war es bisher geplant
     IF urid_old=urid_akt THEN -- wenn die Umsatzrunde, in der wir bisher waren = aktuelle, dann können wir den alten Datensatz verwerfen, sonst würden wir uns doppelt zählen
        DELETE FROM umsatzplanung_umsatzrunde_plusminus WHERE uplrpm_uplr_id=urid_akt AND uplrpm_aglp_id=new.aglp_id;
        RETURN new;
     END IF;
     --nun den Wert in die Umsatzrunde. Beim ausplanen => *-1 für negativen Wert
     INSERT INTO umsatzplanung_umsatzrunde_plusminus(uplrpm_uplr_id, uplrpm_aglp_id, uplrpm_wert) SELECT urid_akt, new.aglp_id, COALESCE(new.aglp_manwert, ag_netto)*IFTHEN(NOT new.aglp_inumsatz, -1, 1) FROM auftg WHERE ag_id=new.aglp_ag_id;
     --
     RETURN new;
   END $$ LANGUAGE plpgsql;

  CREATE TRIGGER auftgliefplanung__1__b_u_aglp_inumsatz
   BEFORE UPDATE OF aglp_inumsatz
   ON auftgliefplanung
   FOR EACH ROW
   EXECUTE PROCEDURE auftgliefplanung__b_u_aglp_inumsatz();

 CREATE OR REPLACE FUNCTION auftgliefplanung__b_u_wertchange() RETURNS trigger AS $$
    DECLARE urid_new INTEGER;
            I   INTEGER;
            S   VARCHAR(100);
            newWert  NUMERIC(16,4);
            oldWert  NUMERIC(16,4);
   BEGIN
     IF auftgliefplanung__b_u_wertchange_disabled() THEN
        RETURN new;
     END IF;
     --
     IF new.aglp_stk IS NOT NULL THEN
        --
        new.aglp_manwert:=ag_ep_netto*new.aglp_stk FROM auftg WHERE ag_id=new.aglp_ag_id;
        --
        IF new.aglp_stk>=ag_stk_uf1-ag_stkl FROM auftg WHERE ag_id=new.aglp_ag_id THEN--endauslieferung / volle Menge erreicht: gesamtwert offen inkl Abzuschläge!
            new.aglp_manwert:=new.aglp_manwert+COALESCE((SELECT SUM(COALESCE(az_abzubetrag, 0)*az_anz) FROM auftgabzu WHERE az_ag_id=new.aglp_ag_id AND az_bebnr IS NULL), 0);
        END IF;
     END IF;
     --
     SELECT MAX(uplr_id) INTO urid_new FROM umsatzplanung_umsatzrunde WHERE uplr_planyearmonth=new.aglp_planyearmonth;       --da sind wir gerade
     --wenn wir uns in der aktuellen umsatzrunde befinden, dann ist das kein +/-, sondern eine korrektur, wir entfernen also einfach die Datensätze, welche zum aktuellen gehören
     DELETE FROM umsatzplanung_umsatzrunde_plusminus WHERE uplrpm_uplr_id=urid_new AND uplrpm_aglp_id=new.aglp_id;          --Änderung innerhalb gleicher Runde, dann weg mit dem vorherigen
     --
     SELECT uplrpm_wert INTO oldwert FROM umsatzplanung_umsatzrunde_plusminus WHERE uplrpm_aglp_id=new.aglp_id ORDER BY uplrpm_id DESC LIMIT 1;  --wert alt geplant
     --
     If new.aglp_manwert IS NULL THEN
        newWert:=ag_netto FROM auftg WHERE ag_id=new.aglp_ag_id;
     ELSE
        newWert:=new.aglp_manwert;
     END IF;
     --
     IF oldwert=newWert THEN
        RETURN new;
     END IF;
     --
     IF urid_new IS NULL THEN
            RAISE EXCEPTION '%', lang_text(29173) /*'keine Umsatzrunde gestartet'*/;
     END IF;
     --Eintragen in Umsatzplanung
     INSERT INTO umsatzplanung_umsatzrunde_plusminus(uplrpm_uplr_id, uplrpm_aglp_id, uplrpm_wert) VALUES (urid_new, new.aglp_id, newWert-COALESCE(oldWert,0));
     --
     RETURN new;
   END $$ LANGUAGE plpgsql;

   CREATE TRIGGER auftgliefplanung__2__b_u_wertchange
      BEFORE UPDATE OF aglp_stk, aglp_manwert
      ON auftgliefplanung
      FOR EACH ROW
      WHEN (new.aglp_inumsatz)
      EXECUTE PROCEDURE auftgliefplanung__b_u_wertchange();


 --
 CREATE TABLE umsatzplanung_umsatzrunde
     (
      uplr_id               SERIAL PRIMARY KEY,
      uplr_planyearmonth    VARCHAR(6),
      uplr_yearmonth        VARCHAR(8) NOT NULL,
      uplr_name             VARCHAR(30) UNIQUE NOT NULL,
      uprl_txt text
     );

   CREATE OR REPLACE FUNCTION public.umsatzplanung_umsatzrunde__a_d() RETURNS trigger AS $$
    BEGIN
     UPDATE auftgliefplanung SET aglp_planyearmonth=NULL, aglp_inumsatz=FALSE, aglp_manwert=NULL WHERE aglp_inumsatz AND aglp_planyearmonth=old.uplr_planyearmonth AND aglp_id IN (SELECT uplrpm_aglp_id FROM umsatzplanung_umsatzrunde_plusminus WHERE old.uplr_id=uplrpm_uplr_id);
     RETURN old;
    END $$ LANGUAGE plpgsql VOLATILE;

    CREATE TRIGGER umsatzplanung_umsatzrunde__a_d
      AFTER DELETE
      ON umsatzplanung_umsatzrunde
      FOR EACH ROW
      EXECUTE PROCEDURE umsatzplanung_umsatzrunde__a_d();
 --
 CREATE TABLE umsatzplanung_umsatzrunde_plusminus
     (
      uplrpm_id             SERIAL PRIMARY KEY,
      uplrpm_uplr_id        INTEGER NOT NULL REFERENCES umsatzplanung_umsatzrunde ON DELETE CASCADE,
      uplrpm_aglp_id        INTEGER REFERENCES auftgliefplanung ON DELETE CASCADE,
      uplrpm_wert           NUMERIC(16,4),
      uplrpm_txt            text
     );

   CREATE INDEX umsatzplanung_umsatzrunde_plusminus_uplrpm_uplr_id ON umsatzplanung_umsatzrunde_plusminus (uplrpm_uplr_id);
   CREATE INDEX umsatzplanung_umsatzrunde_plusminus_uplrpm_aglp_id ON umsatzplanung_umsatzrunde_plusminus (uplrpm_aglp_id);

   CREATE OR REPLACE FUNCTION umsatzplanung_umsatzrunde_plusminus__b_u()RETURNS trigger AS $$
    BEGIN
     PERFORM auftgliefplanung__b_u_wertchange_disable();
     UPDATE auftgliefplanung SET aglp_manwert=new.uplrpm_wert WHERE aglp_id=new.uplrpm_aglp_id;
     PERFORM auftgliefplanung__b_u_wertchange_enable();
     RETURN new;
    END $$ LANGUAGE plpgsql;

    CREATE TRIGGER umsatzplanung_umsatzrunde_plusminus__b_u
      BEFORE UPDATE OF uplrpm_wert
      ON umsatzplanung_umsatzrunde_plusminus
      FOR EACH ROW
      EXECUTE PROCEDURE umsatzplanung_umsatzrunde_plusminus__b_u();




 --
 CREATE OR REPLACE FUNCTION auftgliefplanung__inumsatzrunde(id INTEGER, planyearmonth VARCHAR) RETURNS VARCHAR(500) AS $$
  DECLARE r RECORD;
          S VARCHAR;
  BEGIN
     
     FOR r IN 
        SELECT uplr_id, uplrpm_wert 
          FROM umsatzplanung_umsatzrunde 
          JOIN umsatzplanung_umsatzrunde_plusminus ON uplrpm_uplr_id = uplr_id AND uplrpm_aglp_id = id 
         WHERE uplr_planyearmonth = coalesce(planyearmonth, uplr_planyearmonth) 
     LOOP
        S := coalesce(S || E'\n', '') || IFTHEN(r.uplrpm_wert < 0, '-', '+') || r.uplr_id || ';';
     END LOOP;
     
     RETURN s;
  END $$ LANGUAGE plpgsql STABLE PARALLEL SAFE;

 CREATE OR REPLACE FUNCTION auftgliefplanung__umsatzrunde_plusminus(id INTEGER, planyearmonth VARCHAR) RETURNS VARCHAR(500) AS $$
  DECLARE r RECORD;
        S VARCHAR;
        z INTEGER;
  BEGIN
     z := 0;
     FOR r IN 
        SELECT uplr_id, uplrpm_wert, aglp_manwert, uplrpm_aglp_id 
          FROM umsatzplanung_umsatzrunde 
          LEFT JOIN umsatzplanung_umsatzrunde_plusminus ON uplrpm_uplr_id = uplr_id AND uplrpm_aglp_id = id 
          LEFT JOIN auftgliefplanung ON aglp_id = uplrpm_aglp_id 
         WHERE uplr_planyearmonth = coalesce(planyearmonth, uplr_planyearmonth) 
         ORDER BY uplr_id 
     LOOP
        IF r.uplrpm_aglp_id IS NOT NULL THEN
            IF z > 0 THEN--ab der 2. Umsatzrunde gibts plusse und minusse
                --RAISE NOTICE '%<->%>;%', REPEAT(IFTHEN(r.uplrpm_wert<0, '-', '+'), z)), s, r.uplrpm_wert;
                S := Trim(coalesce( S || E'\n', '') || repeat(IFTHEN(r.uplrpm_wert < 0, '-', '+'), z));
                IF r.aglp_manwert IS NOT NULL THEN
                   S := S || ' ' || (ABS( ROUND(r.uplrpm_wert,2) ) );
                END IF;
            END IF;
        END IF;
        z := z + 1;
     END LOOP;
     RETURN Trim(s);
  END $$ LANGUAGE plpgsql STABLE PARALLEL SAFE;

 CREATE OR REPLACE FUNCTION auftgliefplanung__b_u_wertchange_disabled() RETURNS boolean AS $$
  DECLARE varname VARCHAR;
  BEGIN
    varname:=current_user||'trg_auftgliefplanung__b_u_wertchange';
    RETURN TSystem.Settings__GetInteger(varname)>0;
  END $$ LANGUAGE plpgsql;

 CREATE OR REPLACE FUNCTION auftgliefplanung__b_u_wertchange_disable() RETURNS void AS $$
  DECLARE varname VARCHAR;
  BEGIN
   IF current_user='syncro' THEN
      RETURN;
   END IF;
   varname:=current_user||'trg_auftgliefplanung__b_u_wertchange';
   PERFORM TSystem.Settings__Set(varname, CAST(TSystem.Settings__GetInteger(varname)+1 AS VARCHAR));
   PERFORM disablemodified();
   RETURN;
  END $$ LANGUAGE plpgsql;

 CREATE OR REPLACE FUNCTION auftgliefplanung__b_u_wertchange_enable() RETURNS void AS $$
  DECLARE varname VARCHAR;
  BEGIN
   IF current_user='syncro' THEN
     RETURN;
   END IF;
   varname:=current_user||'trg_auftgliefplanung__b_u_wertchange';
   PERFORM TSystem.Settings__Set(varname, CAST(TSystem.Settings__GetInteger(varname)-1 AS VARCHAR));
   PERFORM enablemodified();
   RETURN;
  END $$ LANGUAGE plpgsql;

--ENDE: UMSATZPLANUNG
-- Einen Produktionsartikel auf Chargenpflicht Prüfen PHKO #8278
CREATE OR REPLACE FUNCTION tauftg.auftg__check_chargenpflicht(agid integer) RETURNS BOOLEAN AS $BODY$
 BEGIN
  RETURN((SELECT true FROM teinkauf.bestvorschlag__check_chargenpflicht(agid) WHERE /*bvs_selbst_chrg or*/ bvs_kopf_chrg));
 END $BODY$ LANGUAGE plpgsql;
--
